import './Field.styl';
import React from 'react';
import PropTypes from 'prop-types';
import {cn} from '@bem-react/classname';
import {Link} from '@components/Link';

const NATIVE_INPUT_ID_PREFIX = 'field';
const b = cn('Field');

class Field extends React.PureComponent {
    static defaultProps = {
        type: 'text'
    };

    ref = React.createRef();

    constructor(props) {
        super(props);

        this.state = {
            isFocused: false,
            value: ''
        };
    }

    focus = () => {
        if (this.ref.current && this.ref.current.focus) {
            this.ref.current.focus();
        }
    };

    handleChange = (event) => {
        const {onFieldChange} = this.props;
        const {value} = event.target;

        this.setState({value}, () => {
            onFieldChange && onFieldChange(value);
        });
    };

    handleFocus = () => {
        const {type} = this.props;

        if (type === 'checkbox') {
            return;
        }

        this.setState({isFocused: true});
    };

    handleBlur = () => {
        const {type, onFieldBlur} = this.props;

        if (type === 'checkbox') {
            return;
        }

        this.setState({isFocused: false});

        if (typeof onFieldBlur === 'function') {
            onFieldBlur();
        }
    };

    render() {
        const {
            label,
            name,
            type,
            error,
            link,
            options = {},
            isLinkVisible,
            placeholder,
            isLabelHidden,
            passwordWeight,
            hasTick,
            isValid
        } = this.props;
        const {isFocused, value} = this.state;
        const additionalParams = {};
        const hasError = Boolean(error);
        const isFilled = Boolean(value);

        if (type === 'checkbox') {
            additionalParams.checked = value;
        }

        return (
            <div
                className={b('', {
                    [type]: true,
                    focused: isFocused,
                    weight: Boolean(passwordWeight),
                    filled: isFilled,
                    failed: hasError,
                    valid: hasTick && isValid
                })}
            >
                <div className={b('input')}>
                    <input
                        ref={this.ref}
                        type={type}
                        id={`${NATIVE_INPUT_ID_PREFIX}${name}`}
                        name={name}
                        {...options}
                        value={value}
                        onChange={this.handleChange}
                        onFocus={this.handleFocus}
                        onBlur={this.handleBlur}
                        placeholder={isFocused ? placeholder : ''}
                        {...additionalParams}
                    />

                    {!isLabelHidden && (
                        <label className={b('label')} htmlFor={`${NATIVE_INPUT_ID_PREFIX}${name}`}>
                            {label}
                        </label>
                    )}
                </div>

                {hasError && <div className={b('error')} dangerouslySetInnerHTML={{__html: error}} />}

                {isLinkVisible && Boolean(link) && (
                    <div className={b('link')}>
                        <Link url={link.url} onClick={link.onClick} pseudo={link.pseudo || false} theme='normal'>
                            {link.text}
                        </Link>
                    </div>
                )}
            </div>
        );
    }
}

Field.propTypes = {
    passwordWeight: PropTypes.string,
    type: PropTypes.string,
    name: PropTypes.string.isRequired,
    label: PropTypes.string.isRequired,
    value: PropTypes.oneOfType([PropTypes.string, PropTypes.bool]),
    error: PropTypes.string,
    onFieldChange: PropTypes.func,
    onFieldBlur: PropTypes.func,
    isLabelHidden: PropTypes.bool,
    isLinkVisible: PropTypes.bool,
    options: PropTypes.shape({
        autoCorrect: PropTypes.string,
        autoCapitalize: PropTypes.string,
        autoComplete: PropTypes.string
    }),
    link: PropTypes.shape({
        url: PropTypes.string,
        text: PropTypes.string,
        onClick: PropTypes.func,
        pseudo: PropTypes.bool
    }),
    placeholder: PropTypes.string,
    hasTick: PropTypes.bool,
    isValid: PropTypes.bool
};

export {Field};
