const config = require('../configs/current');
const url = require('url');
const locs = require('../loc/social.json');
const utils = require('../lib/node-utils');
const commonSetup = require('../lib/commonSetup.js');
const he = require('he');
const fetch = require('../lib/fetch');
const newBrokerExpFilter = require('../lib/newBrokerExpFilter');
const renderPage = require('../dist/index.server').renderPage;

exports.route = (app) => {
    app.get(config.path + '/redirect', newBrokerExpFilter, redirectEntry)
        .get('/broker/redirect', newBrokerExpFilter, redirectEntry)
        .get(config.path + '/start', newBrokerExpFilter, startEntry)
        .get(config.path + '/authz_in_app/start', newBrokerExpFilter, authzInAppStartEntry)
        .get(config.path + '/*/callback', newBrokerExpFilter, callbackEntry)
        .get(config.path + '/authz_in_app/*/callback', newBrokerExpFilter, callbackEntry)
        .all(config.path + '/*/continue', newBrokerExpFilter, continueEntry)
        .get(config.path + '/authz_in_app/*/continue', newBrokerExpFilter, continueEntry)
        .get(config.path + '/*/retry', newBrokerExpFilter, retryEntry)
        .get(config.path + '/*/bind', newBrokerExpFilter, bindEntry)
        .get(config.path + '/tw_reverse_auth_token', newBrokerExpFilter, reverseEntry)
        .all(config.path + '/bind_by_token', newBrokerExpFilter, bindByTokenEntry)
        .post(config.path + '/authz_in_app/entrust_to_account', newBrokerExpFilter, authzInAppEntrustToAccountEntry);
};

const defaultError = {
    error: {
        message: 'Failed to get data from backend',
        code: 'InternalError'
    }
};

const setup = [
    commonSetup,
    (req, res, next) => {
        res.set({
            'X-Content-Type-Options': 'nosniff',
            'X-Frame-Options': 'Deny',
            'Strict-Transport-Security': 'max-age=31536000'
        });

        next();
    }
];

const clearParams = (req, res, next) => {
    const parseUrl = url.parse(req.url);

    res.locals.queData = {};
    res.locals.url = Object.assign({}, config.url, {
        query: {},
        search: parseUrl.search ? parseUrl.search : '',
        pathname: [parseUrl.pathname.replace(config.path, '')].join('')
    });

    next();
};

const doRequest = (req, res) => {
    const {language} = res.locals;

    res.locals.queData.user_ip = req.headers['x-real-ip'];
    res.locals.queData.frontend_url = [req.headers['x-real-scheme'], '://', req.hostname, config.path].join('');

    const loc = locs[language];
    const token = req.body['token'] || req.query['token'];
    const yandexToken = req.body['yandex_token'] || req.query['yandex_token'];
    const provider = req.body['provider'] || req.query['provider'];
    const experiments = req.query.experiments;

    let pageData = {};

    if (token) {
        res.locals.queData.token = encodeURIComponent(token);
    }

    if (yandexToken) {
        res.locals.queData.token = encodeURIComponent(yandexToken);
    }

    ['track', 'yandexuid', 'Session_id'].forEach((item) => {
        if (req.cookies[item]) {
            res.locals.queData[item] = req.cookies[item];
        }
    });

    ['captcha_sid', 'captcha_answer'].forEach((item) => {
        const param = req.body[item] || req.query[item];

        if (param) {
            res.locals.queData[item] = encodeURIComponent(param);
        }
    });

    let app = encodeURIComponent(provider);

    let appName = '';

    // елси хотим ошибку
    // res.locals.url.query.raise_exception = 'AuthorizationRequiredError';

    if (experiments) {
        res.locals.url.query.experiments = experiments;
    }

    const link = url.format(res.locals.url);

    console.log(
        'INFO broker request',
        '[' + new Date() + ']',
        req.headers['x-real-ip'],
        req.headers['user-agent'],
        link
    );

    fetch.call(link, res.locals.queData).then(({err, response, data}) => {
        if (data) {
            data = JSON.parse(data);

            if (data.cookies) {
                res.setHeader('Set-Cookie', data.cookies);
            }
        }
        if (err || data.error) {
            pageData.screen = 'error';
            pageData.title = loc.common['errror.title'];

            if (err) {
                console.error(
                    'WARN broker error',
                    '[' + new Date() + ']',
                    req.headers['x-real-ip'],
                    req.headers['user-agent'],
                    link,
                    err
                );

                pageData.description = loc.error[utils.errorMsg['InternalError']];
                pageData.errorCode = 500;
            } else if (data.error) {
                console.error(
                    'WARN broker error',
                    '[' + new Date() + ']',
                    req.headers['x-real-ip'],
                    req.headers['user-agent'],
                    link,
                    data
                );

                if (data.passthrough && data.retpath) {
                    console.log(
                        'INFO broker redirect',
                        '[' + new Date() + ']',
                        req.headers['x-real-ip'],
                        req.headers['user-agent'],
                        data.retpath
                    );
                    return res.redirect(data.retpath);
                }

                if (data.provider) {
                    if (!app) {
                        app = data.provider.code || data.provider;
                    }
                    appName = data.provider.name || '';
                }

                pageData.provider = app;
                pageData.description = loc.error[utils.errorMsg[data.error.code]];
                pageData.request_id = data.request_id;

                if (data.retry_url && data.error.code !== 'ProviderUnknownError') {
                    pageData.retry_url = data.retry_url;
                }

                if (data.retpath) {
                    pageData.retpath = data.retpath;
                }

                if (data.error.code === 'UserDeniedError') {
                    pageData.description = pageData.description.replace(
                        '%provider%',
                        '<b>' + (loc.common[appName] || '') + '</b>'
                    );
                }

                if (data.error.code === 'InternalError') {
                    pageData.errorCode = 500;
                }

                if (data.error.code === 'RetpathInvalidError') {
                    pageData.errorCode = 400;
                }
            }

            if (res.skipRender) {
                return res
                    .status((err && 500) || response.statusCode || 500)
                    .json((data && data.error) || defaultError);
            }
        } else {
            if (data.state && data.state === 'bind') {
                pageData.screen = 'bind';

                if (!data.display_name.social) {
                    data.display_name.social = {
                        provider: 'ya'
                    };
                }

                pageData = Object.assign({}, pageData, data);
                const profile = [
                    `<span class="s-profile-inline s-profile-${data.provider.code}">`,
                    '<span class="s-profile-icon"></span>',
                    he.escape(data.profile.name || data.profile.username || data.profile.userid),
                    '</span>'
                ].join('');
                const account = [
                    `<span class="s-profile-inline s-profile-${data.display_name.social.provider}">`,
                    '<span class="s-profile-icon"></span>',
                    he.escape(data.display_name.name),
                    '</span>'
                ].join('');

                pageData.description = loc.greet.attach_profile
                    .replace('%profile%', profile)
                    .replace('%account%', account);

                delete pageData.cookies;
            }

            if (data.state && data.state === 'captcha') {
                pageData.screen = 'captcha';
                pageData.description = loc.common['captcha_label'] || '';
                pageData = Object.assign({}, pageData, data);
            }

            if (data.location) {
                console.log(
                    'INFO broker redirect',
                    '[' + new Date() + ']',
                    req.headers['x-real-ip'],
                    req.headers['user-agent'],
                    data.location
                );
                res.redirect(data.location);
                return;
            }
        }

        if (res.skipRender) {
            res.status(200).json(data);
            return;
        }

        res.status(pageData.errorCode ? pageData.errorCode : 200);
        res.locals.store.app = pageData;
        renderPage(req, res);
    });
};

const getParams = (req, params = []) => {
    const query = {};

    params.forEach((item) => {
        const param = he.escape(req.body[item] || req.query[item] || '');

        if (param) {
            if (item === 'bind') {
                query['require_auth'] = param;
            } else {
                query[item] = param;
            }
        }
    });

    return query;
};

const redirectEntry = [
    setup,
    clearParams,
    (req, res, next) => {
        res.locals.url = Object.assign({}, res.locals.url, {
            pathname: '/redirect',
            query: req.query
        });

        next();
    },
    doRequest
];

const startEntry = [
    setup,
    clearParams,
    (req, res, next) => {
        const query = getParams(req, [
            'application',
            'bind',
            'code_challenge',
            'code_challenge_method',
            'consumer',
            'display',
            'force_prompt',
            'login_hint',
            'passthrough_errors',
            'place',
            'provider',
            'require_auth',
            'retpath',
            'return_brief_profile',
            'scope',
            'user_param',
            'sid'
        ]);

        res.locals.url = Object.assign({}, res.locals.url, {
            pathname: '/start',
            search: null,
            query
        });

        if (!req.query['resize']) {
            return next();
        }

        res.locals.store.app = {
            screen: 'redirector',
            location: url.format({
                protocol: 'https',
                host: req.hostname,
                pathname: req.path,
                query: res.locals.url.query
            }),
            app: encodeURIComponent(res.locals.url.query['provider'] || res.locals.url.query['application'])
        };

        renderPage(req, res);
    },
    doRequest
];

const authzInAppStartEntry = [
    setup,
    clearParams,
    (req, res, next) => {
        const query = getParams(req, [
            'application_name',
            'code_challenge',
            'code_challenge_method',
            'consumer',
            'display',
            'passthrough_errors',
            'place',
            'retpath',
            'user_param',
            'yandex_auth_code'
        ]);

        res.locals.url = Object.assign({}, res.locals.url, {
            pathname: '/authz_in_app/start',
            search: null,
            query
        });

        if (!req.query['resize']) {
            return next();
        }

        res.locals.store.app = {
            screen: 'redirector',
            location: url.format({
                protocol: 'https',
                host: req.hostname,
                pathname: req.path,
                query: res.locals.url.query
            }),
            app: encodeURIComponent(res.locals.url.query['provider'] || res.locals.url.query['application'])
        };

        renderPage(req, res);
    },
    doRequest
];

const callbackEntry = [setup, clearParams, doRequest];
const continueEntry = [setup, clearParams, doRequest];

const reverseEntry = [
    setup,
    clearParams,
    (req, res, next) => {
        const query = getParams(req, ['consumer', 'application', 'provider']);

        res.skipRender = true;
        res.locals.url = Object.assign({}, res.locals.url, {
            pathname: '/tw_reverse_auth_token',
            search: null,
            query
        });

        next();
    },
    doRequest
];

const retryEntry = [
    setup,
    clearParams,
    (req, res, next) => {
        const query = getParams(req, [
            'application',
            'bind',
            'consumer',
            'display',
            'force_prompt',
            'login_hint',
            'place',
            'provider',
            'require_auth',
            'retpath',
            'return_brief_profile',
            'scope',
            'sid'
        ]);

        res.locals.url = Object.assign({}, res.locals.url, {
            search: null,
            query
        });

        next();
    },
    doRequest
];

const bindEntry = [
    setup,
    clearParams,
    (req, res, next) => {
        res.locals.queData = {
            allow: 1
        };

        next();
    },
    doRequest
];

const bindByTokenEntry = [
    setup,
    clearParams,
    (req, res, next) => {
        const query = getParams(req, [
            'provider',
            'application',
            'retpath',
            'place',
            'consumer',
            'ui_language',
            'return_brief_profile',
            'scope'
        ]);
        const queData = getParams(req, ['provider_token', 'provider_token_secret']);

        res.locals.queData = Object.assign({}, res.locals.queData, queData);
        res.locals.url = Object.assign({}, res.locals.url, {
            pathname: '/bind_by_token',
            search: null,
            query
        });

        next();
    },
    doRequest
];

const authzInAppEntrustToAccountEntry = [
    setup,
    clearParams,
    (req, res, next) => {
        const queData = getParams(req, ['task_id', 'token', 'code_verifier']);

        res.skipRender = true;
        res.locals.queData = Object.assign({}, res.locals.queData, queData);
        res.locals.url = Object.assign({}, res.locals.url, {
            pathname: '/authz_in_app/entrust_to_account',
            search: null
        });

        next();
    },
    doRequest
];
