'use strict';

const url = require('url');
const utils = require('putils');
const PView = require('pview');
const Logger = require('plog');
const app = require('./prepare-app');
const config = require('./config/current');
const router = require('./router');
const OAuthApi = require('./api/oauth');
const PassportApi = require('./api/passport');
const ErrorPage = require('./pages/error/ErrorPage');
const Controller = require('./controller/Controller');
const AuthController = require('pcontroller/Authentication');
const AuthError = require('pcontroller/AuthError');
const port = process.env.PORT || process.env.npm_package_config_port || 3000;
const host = '127.0.0.1';
const {SERVICE_ALIASES, serviceTicketsSetup} = require('./lib/tvm');
const dnsGracefulStackSwitch = require('dns-graceful-stack-switch');
const dnscache = require('dnscache');
const externalsBlocks = require('./blocks/_externals/externals');
const yateRuntime = require('yate/lib/runtime');
const newRoutes = require('./routes');

// const metricsKey = Buffer.from('jQm9NqfREKGKRNgHSGLV1A==', 'base64');
const lookup = require('./lib/getLookup.js');

dnsGracefulStackSwitch(6);
dnscache({enable: true});

const criticalErrorMessage = `<html><body>Критическая ошибка, мы уже видим её в логах и займемся в ближайшее время.
<br />Critical error. We already see it in the logs and would handle it shortly.</body></html>
`;

try {
    const logSettings = Logger.configure();

    logSettings
        .minLevel(config.loglevel)
        .setFormatter(logSettings.getFormatterByName('production'))
        .setHandler(logSettings.getHandlerByName('buffering'));

    if (config.env === 'development') {
        logSettings
            .setFormatter(logSettings.getFormatterByName('dev'))
            .setHandler(logSettings.getHandlerByName('console'));
    }
} catch (e) {
    console.error('CRITICAL: failed to configure logger', e.stack || e.toString());
    throw e;
}

try {
    externalsBlocks(yateRuntime); // eslint-disable-line

    utils.csrf.setSalt(config.csrfSalt);
    utils.i18n
        .setAllowedLangs(config.loc.langs)
        .setKeysets(config.loc.keysets)
        .loadFromDir(__dirname + '/loc');

    OAuthApi.setAppPasswordsClientIdMapping(config.appPasswordsClientIdMapping);
    PView.setRenderer(PView.yateRenderer).setTemplatesDir(__dirname + '/templates');

    app.use(
        serviceTicketsSetup,
        function setup(req, res, next) {
            const logId =
                req.header('x-request-id') ||
                [req.header('x-real-ip'), Date.now(), Math.floor(Math.random() * 1e10)].join('.');

            req.controller = new Controller(req, res, logId);
            req.logId = logId;
            req.logger = new Logger(logId, 'oauth', 'app');

            next();
        },
        function getDetectors(req, res, next) {
            req.uatraitsDetector = Controller.expressUatraitsDetector;
            // req.langDetector = Controller.expressLangDetector;

            return next();
        },
        function sessionIdRequest(req, res, next) {
            const {controller, logger, logId, headers, serviceTickets} = req;
            const controllerAuthHandle = controller.getAuth();
            const currentUrl = controller.getUrl();

            req.cookies.sessguard = req.cookies.sessguard || '';

            logger.info('Received %s %s', controller.getMethod().toString().toUpperCase(), currentUrl.href);

            if (!controller.getCookie('yandexuid')) {
                // PASSP-10527: Ставить yandexuid при всех запросах на фронт OAuth, если ее нет
                const yandexuid = controller.generateYandexuid();

                let domainBase = '.yandex.';

                logger.info('Setting yandexuid cookie to %s', yandexuid);

                if (config.env === 'production-team') {
                    // PASSP-12254: yandexuid в оаусе на ятиме
                    domainBase = '.yandex-team.';
                }

                controller.setCookie('yandexuid', yandexuid, {
                    domain: domainBase + controller.getTld(),
                    secure: true,
                    maxAge: 315360000000 // 10 лет в миллисекундах
                });
            }

            if (controller.getHeader('x-real-scheme') === 'http') {
                // PASSP-9374 — Не пускать по http в продакшне
                logger.info('Redirecting to https');
                return controller.redirect(url.format(Object.assign({}, currentUrl, {protocol: 'https'})));
            }

            controllerAuthHandle
                .sessionID(
                    {
                        attributes: [
                            AuthController.BB_ATTRIBUTES.GLOBAL_LOGOUT_TIME, // PASSP-10481
                            AuthController.BB_ATTRIBUTES.ACCOUNT_IS_CORPORATE,
                            // PASSP-11894 , see https://beta.wiki.yandex-team.ru/oauth/iface_api/#userinfo
                            AuthController.BB_ATTRIBUTES.REVOKER_TOKENS,
                            AuthController.BB_ATTRIBUTES.REVOKER_APP_PASSWORDS,
                            AuthController.BB_ATTRIBUTES.REVOKER_WEB_SESSIONS // PASSP-12587
                        ].join(',')
                    },
                    {
                        'X-Ya-Service-Ticket': serviceTickets && serviceTickets[SERVICE_ALIASES.BLACKBOX]
                    }
                )
                .then((bbInfo) => {
                    res.locals.bbInfo = bbInfo;

                    next();
                })
                .catch(function (error) {
                    if (error instanceof AuthError && error.code === 'need_resign') {
                        logger.info().type('errorHandler').write('NEED_RESIGN');
                        return controllerAuthHandle.resign();
                    }

                    if (error instanceof AuthError && error.code === 'wrong_guard') {
                        logger.info().type('errorHandler').write('WRONG_GUARD');
                        return controllerAuthHandle.wrongGuard();
                    }

                    if (error) {
                        logger.error().type('errorHandler').write(error);

                        const errorPage = new ErrorPage(
                            controller,
                            new OAuthApi(logId, headers, controller.getLang(), controllerAuthHandle.getUid())
                        );

                        errorPage.setError(error);

                        return errorPage.open();
                    }
                })
                .catch((error) => {
                    // An error while rendering a generic error page
                    logger.critical().type('fallbackErrorHandler').write(error);
                    res.status(500).send(criticalErrorMessage);
                })
                .finally(() => logger.info('Request handled'));
        },

        function setupApi(req, res, next) {
            const {controller, headers, logId, serviceTickets} = req;

            req.oauthApi = new OAuthApi(logId, headers, controller.getLang(), controller.getAuth().getUid());
            req.passportApi = new PassportApi(
                logId,
                {...headers, 'X-Ya-Service-Ticket': serviceTickets && serviceTickets[SERVICE_ALIASES.PASSPORT_API]},
                controller.getLang()
            );
            next();
        },
        function getRegionId(req, res, next) {
            const {logger} = req;
            const iP = req.get('x-real-ip');

            if (iP) {
                try {
                    res.locals.regionId = lookup.getRegionIdByIp(iP);
                    res.locals.countryId = lookup.getCountryId(res.locals.regionId);
                } catch (err) {
                    logger.error().type('oauth', 'app', 'getRegionId').write(err);
                }
            }

            return next();
        },
        /*
        function getYaExperimentsFlags(req, res, next) {
            if (res.locals.experiments) {
                return next();
            }

            return req.controller
                .getYaExperimentInfoPromise()
                .then((experiments) => {
                    const flags = experiments.flags;
                    const flagsString = (Array.isArray(flags) && flags.join(',')) || '';
                    const boxes = experiments.boxes || [];
                    const encodedBoxes = boxes ? encrypt(boxes.join(';'), metricsKey) : '';

                    res.locals.experiments = {
                        flags,
                        flagsString,
                        boxes,
                        encodedBoxes
                    };

                    return next();
                })
                .catch((err) => {
                    req.logger.warn(err);

                    return next();
                });
        }
        */
        function setEmptyExpsFlags(req, res, next) {
            res.locals.experiments = {
                flags: [],
                flagsString: '',
                boxes: [],
                encodedBoxes: ''
            };

            return next();
        }
    );

    newRoutes(app);

    app.use(function oldPageRoutes(req) {
        const {controller, originalUrl, logger, oauthApi} = req;
        const lang = controller.getLang();
        const uid = controller.getAuth().getUid();

        logger.info('User info. Uid: %s, lang: %s', uid, lang);

        const Page = router.matchPage(originalUrl);
        const page = new Page(controller, oauthApi);

        return page.open();
    });

    app.listen(port, host, function () {
        Logger.info().logId('startup').type('startup').write('Successfully started on %s:%s', host, port);
    }).on('error', function (err) {
        Logger.critical().logId('startup').type('startup').write('Failed to start on %s:%s due to', host, port, err);
    });
} catch (e) {
    Logger.critical().logId('startup').type('startup').write('Startup failed due to', e);
}
