const _ = require('lodash');
const assert = require('assert');
const inherit = require('inherit');
const PField = require('pform/Field');
const POauth = require('papi/OAuth');
const Scope = require('papi/OAuth/models/Scope');
const ScopesCollection = require('papi/OAuth/models/ScopesCollection');

/**
 * @constructor
 * @class ScopesField
 * @extends Field
 */
module.exports = inherit(
    PField,
    {
        __constructor: function (clientId) {
            this.clientId = clientId;
            this._init('scopes')
                .setValue(new this.__self.ScopesCollection())
                .addError(new this.__self.Error('missingvalue', 'field.scopes.error.missingvalue'));
        },

        _parseValue: function (formData) {
            assert(_.isPlainObject(formData), 'FormData should be a hash of POSTed values');

            var name = this.getName();
            var Scope = this.__self.Scope;
            return new this.__self.ScopesCollection(
                _(formData)
                    .filter((value, key) => key.indexOf(name) === 0)
                    .map((id) => new Scope(id))
                    .valueOf()
            );
        },

        isPresent: function (formData) {
            assert(_.isPlainObject(formData), 'FormData should be a hash of POSTed values');

            var name = this.getName();
            return _.any(formData, (value, key) => key.indexOf(name) === 0);
        },

        getJSON() {
            return this._json;
        },

        /**
         * @override
         * @param {ScopesCollection}    scopes      A scopes collection
         */
        setValue: function (scopes) {
            assert(scopes instanceof ScopesCollection);

            this._values.value = scopes;
            return this;
        },

        fetchScopes: function (api) {
            return api.userInfoV2().then((userInfo) => {
                if (this.clientId) {
                    return api.clientInfoVer3(this.clientId, true).then((client) => {
                        return _.merge(userInfo.scopes, client.getExtraScopes());
                    });
                }
                return userInfo.scopes;
            });
        },

        compile: function (lang, api) {
            assert(typeof lang === 'string' && lang.length === 2, 'Lang should be a two-letter language code');
            assert(api instanceof POauth, 'Api should be an instance of oauth/api');

            var currentScopes = this.getValue();
            var compiled = this.__base.apply(this, arguments);

            var ttl = currentScopes.getTtl();
            compiled.ttl = {
                time: ttl,
                isDefined: typeof ttl === 'number',
                refreshable: currentScopes.isTtlRefreshable(),
                NOT_DEFINED: Scope.TTL_NOT_DEFINED,
                INFINITE: Scope.TTL_INFINITE
            };

            var id = this.getID();
            var sectionId = 0;

            return this.fetchScopes(api).then((scopes) => {
                const mappedScopes = scopes.map(function (scope) {
                    return {
                        turnedOn: currentScopes.contains(scope),
                        title: scope.getTitle(),
                        id: scope.getId(),
                        section: scope.getSectionTitle(),
                        ttl: scope.getTtl(),
                        ttlRefreshable: scope.isTtlRefreshable(),
                        requiresApproval: scope.requiresApproval()
                    };
                });

                compiled.scopes = mappedScopes.reduce((acc, scope) => {
                    acc[scope.id] = scope;

                    return acc;
                }, {});

                compiled.value = _(mappedScopes)
                    .groupBy((scope) => scope.section)
                    .map(function (permissions, section) {
                        var sectId = id + 'Section' + sectionId++;

                        return {
                            section: section,
                            id: sectId,
                            permissions: _.sortBy(permissions, 'title')
                        };
                    })
                    .sortBy('section')
                    .valueOf();

                return compiled;
            });
        },

        moderationStatus: function (status) {
            this.setOption('moderationStatus', status);
        }
    },
    {ScopesCollection, Scope}
);
