(function() {
    passport.block('scopes', 'control', {
        updateTimeout: 0,

        isRequired: true,
        needsServerValidation: false,
        oldValue: [],

        events: {
            'click .js-scopes-scopes-section': 'showSection',
            'update.scopes': 'onUpdate'
        },

        init: function(options) {
            this._checkboxes = this.$('.js-scopes-permission-checkbox').map(function() {
                return nb.block(this);
            }).get();

            var onChange = this.checkUpdate.bind(this, false);
            this._checkboxes.forEach(function(checkbox) {
                checkbox.on('nb-changed', onChange);
            });

            if (options) {
                this.setInitialModerationStatus(options.moderationStatus);
            }

            this.setInitialScopes(this.val());
            this.updateTokenAndModerationInfo(this.getInitialScopes());

            this._errorPopup = nb.block(this.$('.js-scopes-scopes-errormessages').get(0));
        },

        setInitialModerationStatus: function(status) {
            this._initialStatus = status;
        },
        getInitialModerationStatus: function() {
            return this._initialStatus;
        },

        getScopeData: function(scope) {
            if (this._scopes) {
                return this._scopes[scope];
            }

            try {
                this._scopes = this.$el.data('serialized').scopes;
            } catch(e) {
                this._scopes = {};
            }

            return this.getScopeData(scope);
        },

        /**
         * Scopes, that are present when the page is opened
         * @param {string[]} scopes
         */
        setInitialScopes: function(scopes) {
            this._initialScopes = scopes;
        },

        getInitialScopes: function() {
            return this._initialScopes;
        },

        checkUpdate: function(silent) {
            var currentValue = this.val();

            if (this.oldValue.toString() !== currentValue.toString()) {
                var change = [currentValue, this.oldValue];
                this.oldValue = currentValue;

                this.validate(silent);

                if (!silent) {
                    this.emit('update', change);
                }
            }
        },

        onUpdate: function(event, change) {
            this.updateTokenAndModerationInfo(change[0]);
        },

        updateTokenAndModerationInfo: function(currentValue) {
            //Moderation
            var approvalHiddenClass = 'js-scopes-moderation-requirement_hidden';
            var requiresApproval = this.requiresApproval(currentValue);
            this.$('.js-scopes-moderation-requirement-necessary').toggleClass(approvalHiddenClass, !requiresApproval);
            this.$('.js-scopes-moderation-requirement-unnecessary').toggleClass(approvalHiddenClass, requiresApproval);

            var ttl = this._calculateTtl(currentValue);
            var ttlHiddenClass = 'js-scope-ttl-description_hidden';
            this.$('.js-scope-ttl-description').addClass(ttlHiddenClass);
            if (ttl === 'NOT_DEFINED') {
                this.$('.js-scope-ttl-description-undefined').removeClass(ttlHiddenClass);
            } else if (ttl === 'INFINITE') {
                this.$('.js-scope-ttl-description-infinite').removeClass(ttlHiddenClass);
            } else {
                this.$('.js-scope-ttl-description-defined').removeClass(ttlHiddenClass);
                this.$('.js-scope-ttl-description-refreshable').toggleClass(ttlHiddenClass, !this.isTtlRefreshable(currentValue));
                this.$('.js-scope-ttl-description-time').text(this.spellTime(ttl));
            }
        },

        /**
         * @param {string[]} permissions
         * @private
         */
        _calculateTtl: function(permissions) {
            var that = this;

            if (permissions.length === 0) {
                return 'NOT_DEFINED';
            }

            if (permissions.every(function(scope) {
                return that.getScopeData(scope).ttl === 'INFINITE';
            })) {
                return 'INFINITE';
            }

            return permissions.reduce(function(min, scope) {
                var ttl = Number(that.getScopeData(scope).ttl);

                if (ttl < min) {
                    return ttl;
                }

                return min;
            }, Infinity);
        },

        isTtlRefreshable: function(permissions) {
            //Either scope is not defined or is infinite
            if (typeof this._calculateTtl(permissions) !== 'number') {
                return false;
            }

            //Or the ttl is refreshable if all the scopes have refreshable ttl
            var that = this;
            return permissions.every(function(scope) {
                return that.getScopeData(scope).ttlRefreshable;
            });
        },

        requiresApproval: function(permissions) {
            var that = this;

            if (permissions.every(function(scope) {
                return !that.getScopeData(scope).requiresApproval;
            })) {
                //If all scopes do not require approval, client does not requires approval
                return false;
            }

            if (permissions
                .filter(function(scope) {
                    //Not in an initial set
                    return that.getInitialScopes().indexOf(scope) === -1;
                })
                .some(function(scope) {
                    return that.getScopeData(scope).requiresApproval;
                })) {
                //If any new scope requires approval, the client requires approval
                return true;
            }

            //There is an initial permission that requires approval
            //Respond based on the current status
            return {
                'PENDING': true,
                'REJECTED': true,
                'NOT_REQUIRED': false,
                'RESOLVED': false
            }[this.getInitialModerationStatus()];
        },

        showSection: function(event) {
            var $target = $(event.target);
            var hiddenSectionClass = 'js-scopes-scopes-permissions_hidden';
            var highlightedSectionClass = 'scopes-scopes-section_selected';

            this.$('.js-scopes-scopes-permissions').addClass(hiddenSectionClass)
                .filter('[data-section=' + $target.data('section') + ']')
                .removeClass(hiddenSectionClass);

            this.$('.js-scopes-scopes-section').removeClass(highlightedSectionClass);
            $target.addClass(highlightedSectionClass);
        },

        val: function() {
            if (arguments.length > 0) {
                throw new Error('Setting values not implemented');
            }

            return this._checkboxes
                .map(function(checkbox) {
                    if (checkbox.isChecked()) {
                        return checkbox.getValue();
                    }
                })
                .filter(function(value) {
                    return Boolean(value);
                });
        },

        isEmpty: function() {
            return this.val().length === 0;
        },

        error: function(code) {
            this.parent.error.apply(this, arguments);

            code = Boolean(code);
            var $scopes = this.$('.scopes-scopes');

            $scopes.toggleClass('scopes-scopes-error', code);
            if (code) {
                this._errorPopup.open({
                    where: $scopes.get(0),
                    how: {
                        at: 'right',
                        my: 'left',
                        collision: 'flip flip',
                        autoclose: false,
                        autofocus: false
                    },
                    autoclose: false,
                    appendTo: false,
                    autofocus: false
                });
            } else {
                this._errorPopup.close();
            }
        },

        /**
         * Converts time period in seconds to human readable time
         */
        spellTime: (function() {
            var SECOND = 1;
            var MINUTE = SECOND * 60;
            var HOUR   = MINUTE * 60;
            var DAY    = HOUR * 24;
            var MONTH  = DAY * 30;
            var YEAR   = DAY * 365;

            var localizations = [
                //YEAR
                {
                    one: i18n('%time.year.one'),
                    some: i18n('%time.year.some'),
                    many: i18n('%time.year.many')
                },

                //MONTH
                {
                    one: i18n('%time.month.one'),
                    some: i18n('%time.month.some'),
                    many: i18n('%time.month.many')
                },

                //DAY
                {
                    one: i18n('%time.day.one'),
                    some: i18n('%time.day.some'),
                    many: i18n('%time.day.many')
                },

                //HOUR
                {
                    one: i18n('%time.hour.one'),
                    some: i18n('%time.hour.some'),
                    many: i18n('%time.hour.many')
                },

                //MINUTE
                {
                    one: i18n('%time.minute.one'),
                    some: i18n('%time.minute.some'),
                    many: i18n('%time.minute.many')
                },

                //SECOND
                {
                    one: i18n('%time.second.one'),
                    some: i18n('%time.second.some'),
                    many: i18n('%time.second.many')
                }
            ];

            /**
             * @param {number} time    Time in seconds
             * @returns {string}
             */
            return function(time) {
                if (typeof time !== 'number') {
                    throw new Error('Time should be a number of seconds');
                }

                return [YEAR, MONTH, DAY, HOUR, MINUTE, SECOND]
                    .map(function(period) {
                        var num = Math.floor(time / period);
                        time = time - num * period;
                        return num;
                    })
                    .map(function(num, index) {
                        if (num === 0) {
                            return null;
                        }

                        return num + ' ' + i18n.tanker.dynamic.plural($.extend({ count: num }, localizations[index]));
                    })
                    .filter(function(value) {
                        return Boolean(value);
                    })
                    .join(', ');
            };
        })()
    });
})();
