const assert = require('assert');
const inherit = require('inherit');
const _ = require('lodash');
const PView = require('pview');
const PLog = require('plog');
const PController = require('pcontroller');
const config = require('../../config/current');
const UserView = require('../header/UserView');

const STRIP_TLD_RE = /(\.com\.tr|\.[^.]+)$/;

const UidView = inherit(PView, {
    name: 'UidView',
    __constructor(controller) {
        this.__base.apply(this, arguments);
        this._controller = controller;
    },
    _compile() {
        const controller = this._controller;
        const auth = controller.getAuth();
        const authRequest = auth.getLastRequestPromise() || auth.loggedIn();

        return authRequest
            .then(() => {
                let passportHost;

                switch (config.env) {
                    case 'intranet': {
                        passportHost = 'passport.yandex-team.';
                        break;
                    }
                    case 'production-team': {
                        passportHost = 'passport.yandex-team.';
                        break;
                    }
                    case 'rc-team': {
                        passportHost = 'passport-rc.yandex-team.';
                        break;
                    }
                    case 'testing-team': {
                        passportHost = 'passport-test.yandex-team.';
                        break;
                    }
                    case 'testing': {
                        passportHost = 'passport-test.yandex.';
                        break;
                    }
                    case 'rc': {
                        passportHost = 'passport-rc.yandex.';
                        break;
                    }
                    case 'stress': {
                        passportHost = 'passport-load.yandex.';
                        break;
                    }
                    case 'development': {
                        passportHost = controller.getUrl().hostname.replace(STRIP_TLD_RE, '.');
                        break;
                    }
                    default:
                        passportHost = 'passport.yandex.';
                }
                passportHost += controller.getTld();

                return {
                    ma: {
                        uid: auth.getUid(),
                        login: auth.getLogin(),
                        passportHost
                    }
                };
            })
            .catch((err) => {
                new PLog(controller.getLogId(), 'passport', 'renderinglayout', 'uidview')
                    .warn()
                    .write('authRequest rejected', err);
                return {};
            });
    }
});

module.exports = inherit(require('pview/CompositeView'), {
    template: null,
    header: true,
    footer: true,
    isReactPage: false,

    name: 'LayoutView',
    __constructor(controller) {
        assert(controller instanceof PController, 'Controller should be an instance of PController');

        this.__base.apply(this, arguments);
        this._controller = controller;
    },

    getTemplate() {
        assert(this.template && typeof this.template === 'string', 'Template name should be defined');
        return this.template;
    },

    _compile() {
        const self = this;
        const controller = this._controller;
        const nonce = controller._response.locals.nonce;
        const tld = controller.getTld();
        const pathname = controller.getUrl().pathname;
        const clientPaths = ['/authorize', '/auth/finish'];
        const passportURL = config.paths.passport.replace('%tld%', tld);

        this.append(new UserView(controller));
        this.append(new UidView(controller));

        return Promise.all([this.__base.apply(this, arguments), controller.getCsrfToken()]).then((responses) => {
            const [compiled, csrf] = responses;

            return {
                config: {
                    staticPath: config.paths.static,
                    mdaHost: config.hosts.mda,
                    avatar: config.avatar,
                    origin: controller.getRequestParam('origin'),
                    metrics: {
                        id: this.metricsID || config.metricsID,
                        exp: controller._response.locals.experiments.encodedBoxes
                    },
                    serviceURL: clientPaths.includes(pathname) ? passportURL : '/',
                    nonce
                },
                meta: {
                    logId: controller.getLogId(),
                    env: config.env,
                    template: self.getTemplate().replace(/\.js$/, ''),
                    language: controller.getLang(),
                    domain: controller.getUrl().hostname,
                    tld: controller.getTld(),
                    url: controller.getUrl().href,
                    csrf,
                    year: new Date().getFullYear()
                },
                page: _.assign(compiled, {
                    header: self.header,
                    footer: self.footer,
                    isReactPage: self.isReactPage
                })
            };
        });
    }
});
