(function (global, factory) {
    'use strict';

    if (typeof module === 'object' && module.exports) {
        module.exports = factory();
    } else {
        global.Metrika = factory();
    }
})(this, function () {
    var Metrika = {
        /**
         * @private
         */
        inited: false,

        /**
         * Кэш счётчиков, которые были добавлены, пока метрика ещё не догрузилась
         * @type Object
         * @private
         */
        counts: [],

        /**
         * Кэш целей, которые были добавлены, пока метрика ещё не догрузилась
         * @type Object
         * @private
         */
        goals: [],

        /**
         * Ссылка на счётчик
         * @type Ya.Metrika
         * @private
         */
        counter: null,

        /**
         * Инициализация метрики - загрузка скрипа и созание счётчика
         * @param {String} mcid id счётчика
         */
        init: function (mcid, experiments) {
            if (!mcid) {
                return this.clear();
            }

            var script = document.createElement('script');
            document.body.insertBefore(script, document.body.children[0]);
            script.onload = this.onScriptLoad.bind(this, mcid, experiments);
            script.src = '//mc.yandex.ru/metrika/watch.js';
        },

        onScriptLoad: function (mcid, experiments) {
            /* global Ya */
            if (window.Ya && Ya.Metrika) {
                var settings = {
                    id: mcid,
                    clickmap: true,
                    trackLinks: true,
                    accurateTrackBounce: true,
                    webvisor: true,
                    trackHash: true
                };

                if (experiments) {
                    settings.experiments = experiments;
                }

                try {
                    this.counter = window['yaCounter' + mcid] = new Ya.Metrika(settings);
                } catch (e) {
                    return this.clear();
                }

                // метрика проинициализирована
                this.inited = true;

                // выполняем ранее сохранённые цели
                while (this.counts.length) {
                    this.count.apply(this, this.counts.shift());
                }
                while (this.goals.length) {
                    this.reachGoal.apply(this, this.goals.shift());
                }
            }
        },

        /**
         * Оборачивает переданные параметры в ещё один уровень
         * @param {Object} params
         * @param {String} name - имя обёртки
         * @private
         */
        wrap: function (params, name) {
            var wrapper = {};

            wrapper[name] = params;
            return wrapper;
        },

        /**
         * Сглаживает параметры и фильтрует пустые ячейки
         * @example [["Promo", "Some"], "click"] -> ["Promo", "Some", "click"]
         *
         * @param {Array} input     Данные для сглаживания
         * @returns {Array}
         */
        flatten: function (input) {
            // сглаживаем параметры
            // [["Promo", "Some"], "click"] -> ["Promo", "Some", "click"]
            var flatten = [];
            while (input.length) {
                flatten = flatten.concat(input.shift());
            }

            // удаляем пустые ячейки
            return flatten.filter(function (v) {
                return Boolean(v);
            });
        },

        /**
         * Считает в "Содержание" > "Параметры визитов"
         * @param {Object} params - https://yandex.ru/support/metrika/objects/params-method.xml
         *
         * вместо params можно мередать strig1, string2, string3,...
         * тогда оно склеится в {string1: {string2: string3}}
         * упрощает формирование структуры
         */
        count: function (params) {
            var args = Array.prototype.slice.call(arguments);

            // если метрика не инициализированна -
            // просто сохраним и обработаем позже
            if (!this.inited) {
                return this.counts.push(args);
            }

            args = this.flatten(args);

            // забираем последний параметр
            params = args.pop();

            // если остались ещё параметры - начинаем оборачивания
            while (args.length) {
                params = this.wrap(params, args.pop());
            }

            this.counter.params(params);
        },

        /**
         * Считает в "Отчеты" > "Конверсии"
         * @param {String} target - https://yandex.ru/support/metrika/objects/reachgoal.xml
         *
         */
        reachGoal: function (target) {
            var args = Array.prototype.slice.call(arguments);

            // если метрика не инициализированна -
            // просто сохраним и обработаем позже
            if (!this.inited) {
                return this.goals.push(args);
            }

            this.counter.reachGoal(target);
        },

        /**
         * Нейтрализует метрику
         * @private
         */
        clear: function () {
            this.count = function () {};
            this.counts = [];
            this.goals = [];
        }
    };

    return Metrika;
});
