const PController = require('pcontroller');
const putils = require('putils');
const PLog = require('plog');
const assert = require('assert');
const prequest = require('prequest');
const util = require('util');
const url = require('url');
const expressUaTraits = require('@yandex-int/express-uatraits');
const ClientModel = require('papi/OAuth/models/Client');
const config = require('../config/current');
const {SERVICE_ALIASES} = require('../lib/tvm');

PController.setUtils(putils);

class Controller extends PController {
    constructor(req, res, logId) {
        super(req, res, logId, SERVICE_ALIASES, 'oauth');
        assert(logId, 'LogID should be defined');

        this._logger = new PLog(logId, 'oauth', 'controller');
    }

    getUatraits() {
        if (this._request.uatraits) {
            return Promise.resolve(this._request.uatraits);
        }

        const req = this._request;
        const res = this._response;

        return new Promise((resolve, reject) => {
            this._request.uatraitsDetector(req, res, (error) => {
                if (error) {
                    reject(error);
                } else {
                    if (req.uatraits) {
                        if (req.uatraits.BrowserName === 'MSIE' && req.uatraits.BrowserVersion <= 8) {
                            req.uatraits.isIE = true;
                        }

                        this._request.uatraits = req.uatraits;
                    }

                    resolve(req.uatraits);
                }
            });
        });
    }

    getLang() {
        return (
            this.getLangFromQuery() ||
            {
                ru: 'ru',
                com: 'en',
                'com.tr': 'tr',
                ua: 'uk'
            }[this.getTld()] ||
            'ru'
        );
    }

    getLangFromQuery() {
        const lang = this.getRequestParam('language');

        return (
            {
                ru: 'ru',
                en: 'en',
                tr: 'tr',
                ua: 'uk',
                uk: 'uk'
            }[lang] || null
        );
    }

    getAuth() {
        return super.getAuth().setBlackbox(config.hosts.blackbox);
    }

    getAuthUrl() {
        const authLinks = {
            production: 'https://passport.yandex.%tld%/auth',
            development: 'https://passport-test.yandex.%tld%/auth',
            stress: 'https://passport-load.yandex.%tld%/auth',
            testing: 'https://passport-test.yandex.%tld%/auth',
            'testing-team': 'https://passport-test.yandex-team.ru/auth',
            rc: 'https://passport-rc.yandex.%tld%/auth',
            intranet: 'https://passport.yandex-team.%tld%/auth'
        };

        return url.parse(authLinks[this.getEnv()].replace('%tld%', this.getTld()));
    }

    getAuthSecureUrl() {
        const authUrl = this.getAuthUrl().href;
        const authSecureUrl = url.resolve(authUrl, '/auth/secure');

        return url.parse(authSecureUrl);
    }

    getPaymentAuthRetpath() {
        const currentUrl = url.parse(url.format(this.getUrl()));
        const paymentAuthRetpath = url.resolve(currentUrl, '/authorize/commit');

        return url.parse(paymentAuthRetpath);
    }

    _getAuthVerifyUrl() {
        const authUrl = this.getAuthUrl().href;
        const authVerifyUrl = url.resolve(authUrl, '/auth/verify');

        return url.parse(authVerifyUrl);
    }

    _redirectToClientPage(clientId, postfix) {
        assert(ClientModel.isValidClientId(clientId), 'Client ID should be a 32char long hex');

        this.redirect('/client/' + clientId + postfix);
    }

    redirectToClientPage(clientId) {
        this._redirectToClientPage(clientId, '');
    }

    redirectToClientAndShowSecretUndo(clientId) {
        this._redirectToClientPage(clientId, '/undosecret');
    }

    redirectToClientAndShowSecretUndone(clientId) {
        this._redirectToClientPage(clientId, '/secretundone');
    }

    redirectToCreatedClients() {
        this.redirect('/client/my');
    }

    redirectToIndex() {
        this.redirect('/');
    }

    redirectToVerifyPage(retpath) {
        this.redirect(this.getAuthVerifyUrl(retpath));
    }

    getAuthVerifyUrl(retpath) {
        const authVerifyUrl = this._getAuthVerifyUrl();

        authVerifyUrl.query = {retpath, origin: this.getRequestParam('origin') || 'oauth'};

        return url.format(authVerifyUrl);
    }

    getEnv() {
        if (this.isIntranet()) {
            return 'intranet';
        }

        return process.env.NODE_ENV || 'development';
    }

    getPassportHost() {
        let passportHost;

        switch (config.env) {
            case 'intranet': {
                passportHost = 'passport.yandex-team.';
                break;
            }
            case 'production-team': {
                passportHost = 'passport.yandex-team.';
                break;
            }
            case 'rc-team': {
                passportHost = 'passport-rc.yandex-team.';
                break;
            }
            case 'testing-team': {
                passportHost = 'passport-test.yandex-team.';
                break;
            }
            case 'testing':
            case 'development': {
                passportHost = 'passport-test.yandex.';
                break;
            }
            case 'stress': {
                passportHost = 'passport-load.yandex.';
                break;
            }
            case 'rc': {
                passportHost = 'passport-rc.yandex.';
                break;
            }
            default:
                passportHost = 'passport.yandex.';
        }

        passportHost += this.getTld();

        return passportHost;
    }

    /**
     * Get experiment info
     *
     * @returns {Promise}
     */
    getYaExperimentInfoPromise() {
        const TIMEOUT = 160;

        const logger = this._logger;
        const headers = this.getHeaders();
        const isIntranet = this.isIntranet();
        const userAgent = this.prepareUserAgent(headers['user-agent']);
        const host = headers['host'];
        const regionId = this._response.locals && this._response.locals.regionId;
        const xForwardedFor = headers['x-real-ip'];
        const cookieString = util.format('yandexuid=%s', this.getCookie('yandexuid'));
        const experiments = {
            flags: [],
            boxes: []
        };

        return new Promise((resolve) => {
            const requestHeaders = {
                Cookie: cookieString,
                Host: host,
                'X-Region-City-Id': regionId,
                'User-Agent': userAgent,
                'X-Forwarded-For-Y': xForwardedFor
            };
            const testId = this.getCookie('test-id') || this.getRequestParam('test-id');
            const flagIds = this.getCookie('flag-ids') || this.getRequestParam('flag-ids');

            if (flagIds && (isLocalDev || this.getEnv() === 'development')) {
                experiments.flags = flagIds.split(';');
            }

            let uaasUrl = config.paths.experiments;

            if (isIntranet) {
                return resolve(experiments);
            }

            if (!this.isYandexUidValid()) {
                logger.info('Failed to request exps. Invalid yandexuid: %s', JSON.stringify(requestHeaders));
                return resolve(experiments);
            }

            if (testId) {
                const isYandexInternalNetwork = this.getLocalsField('isYandexInternalNetwork');
                const isProduction = this.getEnv() === 'production';

                if (!isProduction || isYandexInternalNetwork) {
                    requestHeaders['X-Yandex-UAAS'] = 'testing';
                    uaasUrl = url.format(
                        Object.assign({}, url.parse(uaasUrl), {
                            // pathname: null,
                            search: null,
                            query: {
                                'test-id': testId
                            }
                        })
                    );
                }
            }

            prequest({
                url: uaasUrl,
                params: {
                    timeout: TIMEOUT,
                    headers: requestHeaders,
                    responseType: 'text',
                    retry: {
                        calculateDelay: ({attemptCount, retryOptions: {limit = 2}}) =>
                            attemptCount <= limit ? TIMEOUT : 0
                    }
                },
                logID: this.getLogId()
            })
                .then((response = {}) => {
                    const {statusCode, headers = {}} = response;

                    if (statusCode === 200 && headers['x-yandex-expflags']) {
                        logger.info('Experiment status header was: ', headers['x-yandex-expflags']);

                        const experimentBoxes = headers['x-yandex-expboxes'] || '';
                        const flagsHeader = headers['x-yandex-expflags'].split(',');

                        flagsHeader.forEach((header) => {
                            try {
                                const buffer = Buffer.from(header, 'base64');
                                const parsed = JSON.parse(buffer.toString());

                                if (Array.isArray(parsed) && parsed[0] && parsed[0].HANDLER === 'PASSPORT') {
                                    const flags =
                                        (parsed[0].CONTEXT &&
                                            parsed[0].CONTEXT.PASSPORT &&
                                            parsed[0].CONTEXT.PASSPORT.flags) ||
                                        [];

                                    experiments.flags = experiments.flags.concat(flags);
                                }
                            } catch (e) {
                                logger.warn('Reading experiment status was not "ok": ', e);
                            }
                        });

                        experiments.boxes = experimentBoxes.split(';');
                    }

                    return resolve(experiments);
                })
                .catch((error) => {
                    logger.error('Failed to request exps. Headers %s, error %s', JSON.stringify(requestHeaders), error);

                    return resolve(experiments);
                });
        });
    }

    getHeaders() {
        return Object.assign({}, this._request.headers);
    }

    prepareUserAgent(userAgent = '') {
        return userAgent.replace(/\n/g, '');
    }

    isYandexUidValid() {
        const yandexuid = this.getCookie('yandexuid');

        return /^\d+$/.test(yandexuid);
    }

    getLocalsField(field) {
        return this._response.locals[field];
    }

    isIntranet() {
        return process.env.INTRANET === 'intranet';
    }
}

Controller.expressUatraitsDetector = expressUaTraits({
    browser: '/usr/share/uatraits/browser.xml',
    profiles: '/usr/share/uatraits/profiles.xml',
    uatraits: '/usr/lib/node_modules/uatraits'
});

module.exports = Controller;
