const config = require('../../config/current');
const fs = require('fs');
const SERVICE_ALIASES = {
    AFISHA: 'afisha',
    CLOUD: 'cloud',
    ADDRS_SEARCH: 'addrs-search',
    PASSPORT_API: 'passport-api',
    BLACKBOX: 'blackbox',
    BLACKBOX_MIMINO: 'mimino'
};

// promisify
const readFile = (path, options) =>
    new Promise((resolve, reject) => {
        fs.readFile(path, options, (err, data) => {
            if (err) {
                return reject(err);
            }
            return resolve(data);
        });
    });

/**
 * TicketsExtractor asynchronously extracts the tvm-tickets from external file.
 *
 * https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Promise/resolve
 * Thenable
 */

class TicketsExtractor {
    constructor({ticketsFile = config.ticketsFile} = {}) {
        this.ticketsPath = ticketsFile;

        this.ticketsData = null;
        this.serviceTickets = {};

        this.bindAll();
    }

    bindAll() {
        this.readTicketsFile = this.readTicketsFile.bind(this);
        this.parseTickets = this.parseTickets.bind(this);
        this.extractTickets = this.extractTickets.bind(this);
        this.then = this.then.bind(this);
    }

    readTicketsFile() {
        return readFile(this.ticketsPath, 'utf8').then((ticketsData) => {
            this.ticketsData = ticketsData;
        });
    }

    parseTickets() {
        this.parsedTickets = JSON.parse(this.ticketsData);
    }

    extractTickets() {
        const parsedTickets = this.parsedTickets;

        if (!parsedTickets || !parsedTickets.tickets) {
            return;
        }

        const tickets = parsedTickets.tickets;

        Object.keys(tickets).forEach((key) => {
            const ticket = tickets[key];
            const alias = ticket.alias;

            this.serviceTickets[alias] = ticket.ticket;
        });
    }

    get tickets() {
        return Object.assign({}, this.serviceTickets);
    }

    then(resolve, reject) {
        this.readTicketsFile()
            .then(this.parseTickets)
            .then(this.extractTickets)
            .then(() => {
                resolve({tickets: this.tickets});
            })
            .catch(reject);
    }
}

function getTicketsAssoc() {
    return Promise.resolve(new TicketsExtractor()).then(({tickets}) => tickets);
}

const serviceTicketsSetup = (req, res, next) => {
    if (req.hasOwnProperty('serviceTickets')) {
        return next();
    }

    getTicketsAssoc().then((tickets) => {
        req.serviceTickets = tickets;
        next();
    });
};

module.exports = {
    SERVICE_ALIASES,
    TicketsExtractor,
    getTicketsAssoc,
    serviceTicketsSetup
};
