const inherit = require('inherit');
const assert = require('assert');
const Controller = require('../controller/Controller');
const OauthApi = require('papi/OAuth');
const Plog = require('plog');
const MessageView = require('../blocks/message/MessageView');

const AbstractPage = inherit({
    name: null,

    /**
     * @param {Controller} controller
     * @param {Api} api
     * @constructor Page
     * @class Page
     */
    __constructor: function (controller, api) {
        assert(controller instanceof Controller, 'Controller instance expected');
        assert(api instanceof OauthApi, 'Api instance expected');
        assert(this.name && typeof this.name === 'string', 'Page name should be defined for logging');

        this._controller = controller;
        this._api = api;

        this._logger = new Plog(this._controller.getLogId(), 'oauth', 'page', this.name);
    },

    /**
     * This method is called when the page receives control.
     * From now on the system outside does not controls the flow, the page does.
     */
    open: function () {
        throw new Error('Should be overwritten');
    },

    /**
     * Set a message view and open the page
     * @param {MessageView} message
     */
    openWithMessage: function (message) {
        assert(message instanceof MessageView);

        this._messageView = message;
        return this.open();
    },

    _clientNotFoundHandler: function (error) {
        if (!(this instanceof AbstractPage)) {
            throw new Error('Wrong this value, use bind on _clientNotFoundHandler');
        }

        if (error instanceof OauthApi.ApiError && error.contains('client.not_found')) {
            // Нельзя вынести наверх, потому что получится круговая зависимость
            const Page404 = require('./error/Error404Page');
            return new Page404(this._controller, this._api).open();
        }

        throw error;
    }
});

module.exports = AbstractPage;
