const assert = require('assert');
const when = require('when');
const inherit = require('inherit');
const ClientModel = require('papi/OAuth/models/Client');
const PView = require('pview');
const HTTPException = require('../../../lib/exceptions/http/HTTPException');
const ErrorPage = require('../../error/ErrorPage');
const ErrorLayout = require('./ErrorLayout');
const AbstractPage = require('../../../pages/AbstractPage');
const i18n = require('putils').i18n;

const ErrorView = inherit(PView, {
    name: 'auth.errorView',
    __constructor: function (clientModel, platformID, isMagicLink) {
        this.__base.apply(this, arguments);

        const platforms = clientModel.getPlatforms() || {};
        let platform;

        assert(clientModel instanceof ClientModel, 'clientModel should be instance of ClientModel');
        assert(!platform || typeof platform === 'string', 'platform should be a String if defined');

        if (platforms[platformID]) {
            const storeURL = platforms[platformID].appstore_url || '';
            const match = storeURL.match(/id(\d{9})/);
            const appID = (match && match[1]) || '';

            platform = {
                id: platformID,
                appID,
                storeURL
            };
        }

        this._id = clientModel.getId();
        this._title = clientModel.getTitle();
        this._icon = clientModel.getIcon();
        this._platform = platform;
        this._isMagicLink = isMagicLink;
    },

    _compile: function () {
        return {
            id: this._id,
            title: this._title,
            icon: this._icon,
            platform: this._platform,
            isMagicLink: this._isMagicLink
        };
    }
});

/**
 * @extends Page
 */
module.exports = inherit(AbstractPage, {
    name: 'auth.error',

    open: function () {
        const self = this;
        const hostname = this._controller.getUrl().hostname;
        const clientIdMatch = hostname.match(/yx(\w+)/);
        const clientId = clientIdMatch && clientIdMatch[1];

        this._platform = this._controller.getRequestParam('platform');

        this._logger.info('opened');

        if (!clientId) {
            this._logger.info('Client ID is missing');
            return when.reject();
        }

        return this._getClientInfo(clientId)
            .then(function (response) {
                self._render(response);
            })
            .catch(function (error) {
                return new ErrorPage(self._controller, self._api)
                    .setError(
                        new HTTPException(400, i18n(self._controller.getLang(), 'authorize.errors.client_id_unknown'))
                    )
                    .open();
            });
    },

    _getClientInfo(clientId) {
        return when.resolve().then(() => {
            return this._api.clientInfoVer3(clientId);
        });
    },

    /**
     * Render the form
     * @param {ClientModel} clientModel
     * @private
     */
    _render: function (clientModel) {
        assert(clientModel instanceof ClientModel, 'clientModel should be instance of ClientModel');

        const self = this;
        const isMagicLink = this._controller.getUrl().pathname.startsWith('/magic-link');
        const layout = new ErrorLayout(this._controller);
        const errorView = new ErrorView(clientModel, this._platform, isMagicLink);

        this._logger.debug('Rendering');
        return layout
            .append(errorView)
            .render(this._controller.getLang())
            .then(function (rendered) {
                self._logger.verbose('Sending', rendered);
                return self._controller.sendPage(rendered);
            })
            .then(function () {
                self._logger.info('Page sent');
            })
            .catch(function (error) {
                self._logger.error('Rendering error', error);

                return new ErrorPage(self._controller, self._api)
                    .setError(new HTTPException(500, i18n(self._controller.getLang(), 'authorize.errors.internal')))
                    .open();
            });
    }
});
