const PView = require('pview');
const url = require('url');
const inherit = require('inherit');
const config = require('../../config/current');
const {SERVICE_ALIASES} = require('../../lib/tvm');
const langs = config.loc.langs;
const reactTemplates = {};
const reactPage = 'authorize';

const HAS_PLUS_ATTRIBUTE = 1015;

/* eslint-disable global-require */
if (process.env.NODE_ENV !== 'development') {
    langs.forEach((lang) => {
        reactTemplates[lang] = require(`../../templates/${reactPage}.react.${lang}.js`).default;
    });
}

const mapScope = (scope) => ({
    id: scope.getId(),
    title: scope.getTitle(),
    optional: scope.getOptional(),
    alreadyGranted: scope.getAlreadyGranted()
});

const sortScopes = (scopes) =>
    [...scopes]
        .sort((scope, other) => (scope.id <= other.id ? 1 : -1))
        .sort((scope, other) => (scope.alreadyGranted === other.alreadyGranted ? 0 : scope.alreadyGranted ? -1 : 1));

const mapScopes = (scopes) => {
    const bySection = scopes.length > 3;

    if (bySection) {
        const sections = scopes.groupBy((scope) => scope.getSectionTitle());

        return Object.keys(sections).reduce(
            (acc, section) => [...acc, ...sortScopes(sections[section].map(mapScope))],
            []
        );
    }

    return sortScopes(scopes.map(mapScope));
};

module.exports = inherit(PView, {
    name: 'initial',
    __constructor: function (controller, api, requestId, scopes, redirectUri, responseType, state) {
        this.__base.apply(this, arguments);

        this._controller = controller;
        this._requestId = requestId;
        this._redirectUri = redirectUri;
        this._state = state;
        this._responseType = responseType;
        this._api = api;
        this._scopes = scopes;
    },

    _compile: function () {
        const controller = this._controller;
        const urlObj = controller.getUrl();
        const retpath = url.format(urlObj);
        const req = controller._request;

        const clientId = controller.getRequestParam('client_id');

        return Promise.all([
            this._api.clientInfoVer3(clientId),
            controller
                .getAuth()
                .sessionID(
                    {attributes: [HAS_PLUS_ATTRIBUTE].join(',')},
                    {'X-Ya-Service-Ticket': req.serviceTickets && req.serviceTickets[SERVICE_ALIASES.BLACKBOX]}
                ),
            controller.getCsrfToken()
        ]).then(([client, sessionInfo, csrf]) => {
            const displayName = (sessionInfo && sessionInfo.display_name) || {};
            const formValues = {
                retpath,
                clientId,
                csrf,
                device_id: controller.getRequestParam('device_id'),
                device_name: controller.getRequestParam('device_name'),
                display: controller.getRequestParam('display'),
                login_hint: controller.getRequestParam('login_hint'),
                scope: controller.getRequestParam('scope'),
                optional_scope: controller.getRequestParam('optional_scope'),
                request_id: this._requestId,
                redirect_uri: this._redirectUri,
                state: this._state,
                response_type: this._responseType
            };
            const store = {
                authorize: {
                    client: {
                        clientTitle: client.getTitle(),
                        clientIcon: (client.getIcon() || '').replace('/normal', '/big'),
                        isYandex: client.isYandex()
                    },
                    form: Object.keys(formValues).map((field) => ({field, value: formValues[field]})),
                    account: sessionInfo
                        ? {
                              displayName: displayName.name,
                              hasPlus: Boolean(sessionInfo.attributes && sessionInfo.attributes[HAS_PLUS_ATTRIBUTE]),
                              avatarPath: (config.mds || '')
                                  .replace('/islands-200', '/islands-retina-middle')
                                  .replace(
                                      '%avatarId%',
                                      (displayName && displayName.avatar && displayName.avatar.default) || '0/0-0'
                                  ),
                              login: sessionInfo.login,
                              changeAccountLink: url.format(
                                  Object.assign({}, controller.getAuthUrl(), {
                                      query: {retpath, origin: controller.getRequestParam('origin') || 'oauth'}
                                  })
                              ),
                              lang: controller.getLang(),
                              uid: controller.getAuth().getUid()
                          }
                        : {},
                    scopes: mapScopes(this._scopes),
                    bgNumber: Math.floor(new Date().getTime() / 1000 / 60 / 10) % 18
                },
                common: {
                    yandexuid: controller.getCookie('yandexuid'),
                    csrf
                },
                metrics: {
                    header: 'Редизайн /authorize',
                    experiments: controller._response.locals.experiments.encodedBoxes || ''
                }
            };

            const lang = controller.getLang();

            if (process.env.NODE_ENV === 'development') {
                reactTemplates[lang] = require(`../../templates/${reactPage}.react.${lang}.js`).default;
            }

            return {
                reactMarkup: reactTemplates[lang](store),
                reduxState: JSON.stringify(store)
            };
        });
    }
});
