/**
 * What user sees as a reason for a redirect
 * @extends View
 */
var AuthErrorView = require('inherit')(require('pview'), {
    name: 'authorize.authError',
    __constructor: function(code, description) {
        require('assert')(code && typeof code === 'string', 'Error code should be a string');
        require('assert')(description && typeof description === 'string', 'Error description should be a string');

        this.__base.apply(this, arguments);

        this._code = code;
        this._description = description;
    },

    _compile: function() {
        return { authError: {
            code: this._code,
            description: this._description
        } };
    }
});

/**
 * An url view
 * @extends View
 */
var RedirectUrlView = require('inherit')(require('pview'), {
    name: 'authorize.redirectUrl',
    __constructor: function(url) {
        require('assert')(url && typeof url === 'string', 'Url should be a string');
        this.__base.apply(this, arguments);
        this._url = url;
    },

    _compile: function() {
        return { redirectUrl: this._url };
    }
});

/**
 * @class OutboundRedirectPage
 * @typedef OutboundRedirectPage
 * @extends Page
 */
module.exports = require('inherit')(require('../../../pages/AbstractPage'), {
    name: 'authorize.outboundRedirect',

    open: function() {
        require('assert')(this._url && this._errorCode, 'Url and error code should be defined with .url() and .errorCode() methods');

        this._logger.info('opened for url %s with errorCode %s and errorDescription %s', this._url, this._errorCode, this._errorDescription);

        var Layout = require('./OutboundRedirectLayout');
        var layout = new Layout(this._controller);

        var that = this;
        this._logger.debug('Rendering');
        return layout
            .append(new AuthErrorView(this._errorCode, this._errorDescription))
            .append(new RedirectUrlView(this._url))
            .render()
            .then(function(rendered) {
                that._logger.verbose('Sending', rendered);
                return that._controller.sendPage(rendered);
            })
            .then(function() {
                that._logger.info('Page sent');
            });
    },

    /**
     * Url setter
     * @param {string} url
     * @returns {OutboundRedirectPage}
     */
    url: function(url) {
        require('assert')(url && typeof url === 'string', 'Url should be a string');
        this._url = url;
        return this;
    },

    /**
     * Error code setter
     * @param {string} code
     * @returns {OutboundRedirectPage}
     */
    errorCode: function(code) {
        require('assert')(code && typeof code === 'string', 'Error code should be a string');
        this._errorCode = code;
        return this;
    },

    /**
     * Error description setter
     * @param {string} description
     * @returns {OutboundRedirectPage}
     */
    errorDescription: function(description) {
        require('assert')(description && typeof description === 'string', 'Error description should be a string');
        this._errorDescription = description;
        return this;
    }
});
