const when = require('when');
const inherit = require('inherit');
const {ApiError} = require('papi/OAuth');
const i18n = require('putils').i18n;
const AbstractPage = require('../AbstractPage');
const Message = require('../../blocks/message/MessageView');
const CreatedClients = require('./my/MyPage');
const HTTPException = require('../../lib/exceptions/http/HTTPException');
const ErrorPage = require('../error/ErrorPage');

module.exports = inherit(AbstractPage, {
    name: 'client.delete',

    open: async function () {
        this._logger.info('opened');

        const self = this;
        const clientId = this._controller.getUrl().pathname.replace('/client/delete/', '');

        let csrfValid;

        try {
            csrfValid = await this._controller.isCsrfTokenValidV2();
        } catch {
            csrfValid = false;
        }

        if (!csrfValid) {
            this._logger.info('Failed attempt to delete client %s', clientId);
            this._controller.redirectToCreatedClients();
            return when.reject();
        }

        return this._api
            .clientInfoVer3(clientId)
            .then(function (client) {
                return self._api.deleteClientV2(clientId).then(function () {
                    return new CreatedClients(self._controller, self._api).openWithMessage(
                        new Message('app.deleted', client.getTitle())
                    );
                });
            })
            .catch(self._notFoundHandler.bind(self))
            .catch(self._notEditableHandler.bind(self))
            .catch(self._passwordRequiredHandler.bind(self));
    },

    _show403: function () {
        return new ErrorPage(this._controller, this._api)
            .setError(new HTTPException(403, i18n(this._controller.getLang(), 'common.errors.access_denied')))
            .open();
    },

    _passwordRequiredHandler: function (error) {
        if (error instanceof ApiError && error.contains('password.required')) {
            const retPath = this._controller.getFormData().retpath;
            return this._controller.redirectToVerifyPage(retPath);
        }

        throw error;
    },

    _notFoundHandler: function (error) {
        if (error instanceof ApiError && error.contains('client.not_found')) {
            return this._controller.redirectToCreatedClients();
        }

        throw error;
    },

    _notEditableHandler: function (error) {
        if (error instanceof ApiError && error.contains('client.not_editable')) {
            return this._show403();
        }

        throw error;
    }
});
