const inherit = require('inherit');
const {ApiError} = require('papi/OAuth');
const i18n = require('putils').i18n;
const AbstractPage = require('../AbstractPage');
const HTTPException = require('../../lib/exceptions/http/HTTPException');
const ErrorPage = require('../error/ErrorPage');

module.exports = inherit(AbstractPage, {
    name: 'client.resetPassword',

    _clientRe: /\/client\/password\/reset(?:\/undo)?\/(.+)/,
    open: async function () {
        this._logger.info('opened');

        let pathname = this._controller.getUrl().pathname;
        const clientId = pathname.match(this._clientRe)[1];

        this._logger.info('Client id: %s', clientId);

        const that = this;

        let csrfValid;

        try {
            csrfValid = await this._controller.isCsrfTokenValidV2();
        } catch {
            csrfValid = false;
        }

        if (!csrfValid) {
            that._logger.info('Failed attempt at password manipulations for client %s', clientId);
            that._controller.redirectToClientPage(clientId);
            return require('when').reject();
        }

        pathname = pathname.replace(clientId, '');
        if (pathname === '/client/password/reset/') {
            that._logger.info('Resetting');
            return that.reset(clientId);
        } else {
            that._logger.info('Undoing');
            return that.undo(clientId);
        }
    },

    /**
     * Reset the secret for the given client
     * @param {string} clientId
     * @private
     */
    reset: function (clientId) {
        const that = this;
        return this._api
            .newClientSecretV2(clientId)
            .then(function () {
                return that._controller.redirectToClientAndShowSecretUndo(clientId);
            })
            .catch(this._clientNotFoundHandler.bind(this))
            .catch(this._passwordRequiredHandler.bind(this));
    },

    /**
     * Undo resetting the secret for the given client
     * @param {string} clientId
     * @private
     */
    undo: function (clientId) {
        return this._api
            .undoNewClientSecretV2(clientId)
            .then(() => this._controller.redirectToClientAndShowSecretUndone(clientId))
            .catch(this._clientNotFoundHandler.bind(this))
            .catch(this._passwordRequiredHandler.bind(this));
    },

    _show403: function () {
        return new ErrorPage(this._controller, this._api)
            .setError(new HTTPException(403, i18n(this._controller.getLang(), 'common.errors.access_denied')))
            .open();
    },

    _passwordRequiredHandler: function (error) {
        if (error instanceof ApiError && error.contains('password.required')) {
            const retPath = this._controller.getFormData().retpath;
            return this._controller.redirectToVerifyPage(retPath);
        }

        throw error;
    },

    _notEditableHandler: function (error) {
        if (error instanceof ApiError && error.contains('client.not_editable')) {
            return this._show403();
        }

        throw error;
    }
});
