const inherit = require('inherit');
const {ApiError} = require('papi/OAuth');
const i18n = require('putils').i18n;
const AbstractPage = require('../AbstractPage');
const HTTPException = require('../../lib/exceptions/http/HTTPException');
const ErrorPage = require('../error/ErrorPage');
const Message = require('../../blocks/message/MessageView');
const Index = require('../index/IndexPage');

module.exports = inherit(AbstractPage, {
    name: 'client.revoke',

    open: async function () {
        this._logger.info('opened');

        const clientId = this._controller.getUrl().pathname.replace('/client/revoke/', '');

        const that = this;

        let csrfValid;

        try {
            csrfValid = await this._controller.isCsrfTokenValidV2();
        } catch {
            csrfValid = false;
        }

        if (!csrfValid) {
            that._logger.info('Failed attempt to revoke access to client %s', clientId);
            that._controller.redirectToIndex();
            return require('when').reject();
        }

        return that._api
            .clientInfoVer3(clientId)
            .then(function (client) {
                return that._api.gLogoutClient(clientId).then(function () {
                    return new Index(that._controller, that._api).openWithMessage(
                        new Message('app.revoked', client.getTitle())
                    );
                });
            })
            .catch(that._notFoundHandler.bind(that))
            .catch(that._notEditableHandler.bind(that))
            .catch(that._passwordRequiredHandler.bind(that));
    },

    _show403: function () {
        return new ErrorPage(this._controller, this._api)
            .setError(new HTTPException(403, i18n(this._controller.getLang(), 'common.errors.access_denied')))
            .open();
    },

    _passwordRequiredHandler: function (error) {
        if (error instanceof ApiError && error.contains('password.required')) {
            return this._controller.redirectToVerifyPage(this._controller.getFormData().retpath);
        }

        throw error;
    },

    _notFoundHandler: function (error) {
        if (error instanceof ApiError && error.contains('client.not_found')) {
            return this._controller.redirectToIndex();
        }

        throw error;
    },

    _notEditableHandler: function (error) {
        if (error instanceof ApiError && error.contains('client.not_editable')) {
            return this._show403();
        }

        throw error;
    }
});
