'use strict';

const assert = require('assert');
const _ = require('lodash');
const url = require('url');
const when = require('when');
const inherit = require('inherit');
const AuthController = require('pcontroller/Authentication');
const OAuth = require('papi/OAuth');
const Pform = require('pform');
const putils = require('putils');
const ClientModel = require('papi/OAuth/models/Client');
const FormView = require('./FormView');
const SettingsView = require('./SettingsView');
const EditLayout = require('./EditLayout');
const AbstractPage = require('../../AbstractPage');
const HTTPException = require('../../../lib/exceptions/http/HTTPException');
const ErrorPage = require('../../error/ErrorPage');
const ApiError = OAuth.ApiError;

module.exports = inherit(AbstractPage, {
    name: 'client.edit',

    open: function () {
        if (!this._controller.getAuth().isLoggedIn()) {
            this._logger.info('User not logged in, redirecting to auth');
            this._controller.getAuth().authorize({noreturn: 1});
            return when.reject();
        }

        if (this._controller.getAuth().isAutologged()) {
            this._logger.info('User logged in with autologin, redirecting to lightauth2full');
            this._controller.getAuth().lightauth2full(true);
            return when.reject();
        }

        this._logger.info('opened');
        const url = this._controller.getUrl().pathname.replace(/\/+$/, '');
        const method = this._controller.getMethod();
        const params = url.match(/^\/client\/(new|edit|validate)\/?([0-9a-f]{32})?$/) || [];
        const action = params[1];
        const clientId = params[2];

        if (method.isGet()) {
            if (clientId) {
                return this._editExisting(clientId);
            } else {
                return this._createNew();
            }
        }

        if (method.isPost()) {
            switch (action) {
                case 'new': {
                    return this._save();
                }
                case 'edit': {
                    return this._save(clientId);
                }
                case 'validate': {
                    return this._validate(clientId);
                }
            }
        }

        return this._createNew();
    },

    _isUserCorporate: function () {
        return this._controller.getAuth().getAttribute(AuthController.BB_ATTRIBUTES.ACCOUNT_IS_CORPORATE) === '1';
    },

    _getForm: function (clientId) {
        if (this._form) {
            return this._form;
        }

        return (this._form = new Pform(
            new (require('../../../blocks/field/csrf/csrf.field'))(this._controller),
            new (require('../../../blocks/field/title/title.field'))().setRequired(),
            new (require('../../../blocks/field/yandexAppFlag/yandexAppFlag.field'))().setUserCorporate(
                this._isUserCorporate()
            ),
            new (require('../../../blocks/field/description/description.field'))(),
            new (require('../../../blocks/field/icon/icon.field'))(),
            new (require('../../../blocks/field/url/homepage/homepage.field'))(),
            new (require('../../../blocks/field/scopes/scopes.field'))(clientId).setRequired(),
            new (require('../../../blocks/field/url/callback/callback.field'))(),
            new (require('../../../blocks/field/submit/submit.field'))()
        ).setApi(this._api));
    },

    /**
     * Fill the form with values from the Client
     *
     * @param {Form} form
     * @param {Client} client
     * @returns {Form}
     * @private
     */
    _setFormValues: function (form, client) {
        assert(form instanceof Pform, 'form should be an instance of pform');
        assert(client instanceof ClientModel, 'Client should be an client model');

        form.getField('title').setValue(client.getTitle());
        form.getField('description').setValue(client.getDescription());
        form.getField('homepage').setValue(client.getHomepage());
        form.getField('callback').setValue(client.getCallbackUrl());
        form.getField('yandexAppFlag').setValue(client.isYandex() ? 'on' : 'off');
        form.getField('icon').setValue(client.getIconId()).setOption('isYandex', client.isYandex());

        const scopesField = form.getField('scopes');
        scopesField.setValue(client.getScopes());
        scopesField.setValue(client.getExtraScopes());
        scopesField.moderationStatus(client.getApprovalStatus());

        return form;
    },

    /**
     * Create a new client
     * @private
     */
    _createNew: function () {
        return this._render(new FormView(this._getForm(), false, {}, this._controller));
    },

    /**
     * Edit existing client given by id
     * @param {string} clientId
     * @private
     */
    _editExisting: function (clientId) {
        return this._api.clientInfoVer3(clientId, true).then((client) => {
            if (!client.isViewedByOwner()) {
                return this._show403();
            }
            return this._render(
                new FormView(
                    this._setFormValues(this._getForm(clientId), client),
                    clientId,
                    client.getJSONResponse(),
                    this._controller
                )
            );
        }, this._clientNotFoundHandler.bind(this));
    },

    /**
     * Save the client
     * If client id is given, an existing client is amended
     * If no id given, a new client is created
     *
     * @param {string} [clientId]
     * @private
     */
    _save: async function (clientId) {
        const self = this;
        const logger = this._logger;
        const controller = this._controller;
        const csrf = controller.getRequestParam('csrf');

        if (!csrf) {
            logger.error('CSRF-token is required');
            return controller._response.json({
                status: 'error',
                errors: ['form.internal']
            });
        }

        let formData;

        try {
            formData = JSON.parse(controller.getFormData().data);
        } catch (e) {
            logger.error('JSON parsing error');
            return controller._response.json({
                status: 'error'
            });
        }

        const iconFile = controller.getFormData().icon;
        const formArgs = {
            clientId,
            title: formData.title,
            description: formData.description,
            homepage: formData.homepage,
            isYandex: formData.is_yandex,
            iconId: formData.icon_id,
            iconFile,
            platforms: formData.platforms,
            iosAppId: formData.ios_app_id,
            iosAppstoreUrl: formData.ios_appstore_url,
            androidPackageName: formData.android_package_name,
            androidAppstoreUrl: formData.android_appstore_url,
            androidCertFingerprints: formData.android_cert_fingerprints,
            redirectUris: formData.redirect_uri,
            turboappBaseUrl: formData.turboapp_base_url,
            scopes: formData.scopes,
            ownerUids: formData.owner_uids,
            ownerGroups: formData.owner_groups
        };

        let csrfValid;

        try {
            csrfValid = await controller.isCsrfTokenValidV2();
        } catch {
            csrfValid = false;
        }

        if (!csrfValid) {
            logger.error('CSRF-token is not valid');
            return controller._response.json({
                status: 'error',
                errors: ['form.internal']
            });
        }

        let action;

        if (clientId) {
            logger.info('Saving client %s with %j', clientId, formData);
            action = self._api.editClientV3(formArgs);
        } else {
            logger.info('Creating new client with %j', formData);
            action = self._api.createClientV3(formArgs);
        }

        return action
            .then(function (response) {
                logger.info('Client %s saved successfully', response.client_id);
                return controller._response.json(response);
            })
            .catch(self._notSecureSessionHandler.bind(self))
            .catch(self._clientNotFoundHandler.bind(self))
            .catch(self._creatorRequiredHandler.bind(self))
            .catch(self._notEditableHandler.bind(self))
            .catch(function (error) {
                if (error instanceof ApiError) {
                    return controller._response.json(error._response);
                } else {
                    logger.info('Not an API error, gets rethrown: %s', error);
                    throw error;
                }
            });
    },
    /**
     * Validate application changes
     *
     * @param {string} [clientId]
     * @private
     */
    _validate: async function (clientId) {
        const self = this;
        const logger = this._logger;
        const controller = this._controller;
        const csrf = controller.getRequestParam('csrf');

        if (!csrf) {
            logger.error('CSRF-token is required');
            return controller._response.json({
                status: 'error',
                errors: ['form.internal']
            });
        }

        let formData;

        try {
            formData = JSON.parse(controller.getFormData().data);
        } catch (e) {
            logger.error('JSON parsing error');
            return controller._response.json({
                status: 'error'
            });
        }

        const iconFile = controller.getFormData().icon;
        const formArgs = {
            clientId,
            title: formData.title,
            description: formData.description,
            homepage: formData.homepage,
            isYandex: formData.is_yandex,
            iconId: formData.icon_id,
            iconFile,
            platforms: formData.platforms,
            iosAppId: formData.ios_app_id,
            iosAppstoreUrl: formData.ios_appstore_url,
            androidPackageName: formData.android_package_name,
            androidAppstoreUrl: formData.android_appstore_url,
            androidCertFingerprints: formData.android_cert_fingerprints,
            redirectUris: formData.redirect_uri,
            turboappBaseUrl: formData.turboapp_base_url,
            scopes: formData.scopes,
            ownerUids: formData.owner_uids,
            ownerGroups: formData.owner_groups
        };

        logger.info('Validating client %s with %j', clientId, formData);

        let csrfValid;

        try {
            csrfValid = await controller.isCsrfTokenValidV2();
        } catch {
            csrfValid = false;
        }

        if (!csrfValid) {
            logger.error('CSRF-token is not valid');
            return controller._response.json({
                status: 'error',
                errors: ['form.internal']
            });
        }

        return self._api
            .validateClientChangesV3(formArgs)
            .then(function (response) {
                logger.info('Client %s validated successfully', response.client_id);
                controller._response.json(response);
            })
            .catch(self._notSecureSessionHandler.bind(self))
            .catch(self._clientNotFoundHandler.bind(self))
            .catch(self._creatorRequiredHandler.bind(self))
            .catch(self._notEditableHandler.bind(self))
            .catch(function (error) {
                if (error instanceof ApiError) {
                    return controller._response.json(error._response);
                } else {
                    logger.info('Not an API error, gets rethrown: %s', error);
                    throw error;
                }
            });
    },

    /**
     * Render the form
     * @param {FormView} formView
     * @private
     */
    _render: function (formView) {
        assert(formView instanceof FormView, 'Given view should be an instance of FormView');
        const layout = new EditLayout(this._controller);

        this._logger.debug('Rendering');
        return layout
            .append(formView)
            .append(new SettingsView())
            .render(this._controller.getLang())
            .then((rendered) => {
                this._logger.verbose('Sending', rendered);
                return this._controller.sendPage(rendered);
            })
            .then(() => this._logger.info('Page sent'));
    },

    _show403: function () {
        return new ErrorPage(this._controller, this._api)
            .setError(new HTTPException(403, putils.i18n(this._controller.getLang(), 'common.errors.access_denied')))
            .open();
    },

    _creatorRequiredHandler: function (error) {
        if (error instanceof ApiError && error.contains('client.creator_required')) {
            return this._show403();
        }

        // Unexpected error should be rethrown
        throw error;
    },

    _notEditableHandler: function (error) {
        if (error instanceof ApiError && error.contains('client.not_editable')) {
            return this._show403();
        }

        // Unexpected error should be rethrown
        throw error;
    },

    _notSecureSessionHandler: function (error) {
        if (error instanceof ApiError && error.contains('sslsession.required')) {
            const controller = this._controller;
            const retPath = url.format(controller.getUrl());
            return controller.redirectToVerifyPage(retPath);
        }

        // Unexpected error should be rethrown
        throw error;
    }
});
