'use strict';

/**
 * @typedef FormView
 * @class FormView
 * @extends View
 */
const assert = require('assert');
const PView = require('pview');
const PForm = require('pform');
const url = require('url');
const inherit = require('inherit');
const ClientModel = require('papi/OAuth/models/Client');
const config = require('../../../config/current');
const langs = config.loc.langs;
const reactTemplates = {};
const reactPage = 'client.edit';

/* eslint-disable global-require */
if (process.env.NODE_ENV !== 'development') {
    langs.forEach(function (lang) {
        reactTemplates[lang] = require(`../../../templates/${reactPage}.react.${lang}.js`).default;
    });
}

module.exports = inherit(PView, {
    name: 'client.formView',
    __constructor: function (form, clientId, json, controller) {
        const yandexAppField = form.getField('yandexAppFlag');
        const isCorporate = yandexAppField.isCorporate();

        this.__base.apply(this, arguments);

        assert(form instanceof PForm, 'Form should be an instance of pform');
        assert(!clientId || ClientModel.isValidClientId(clientId), 'Client ID should be valid if defined');

        this._form = form;
        this._clientId = clientId;
        this._controller = controller;
        this._store = {
            editPage: {
                retpath: url.format(controller.getUrl()),
                clientId,
                allScopes: '',
                application: json.client,
                isCorporate,
                viewed_by_owner: json.viewed_by_owner,
                can_be_edited: json.can_be_edited
            },
            metrics: {
                header: 'oauth/client',
                experiments: controller._response.locals.experiments.encodedBoxes || ''
            }
        };
    },

    _compile: function (lang) {
        assert(typeof lang === 'string' && lang.length === 2, 'Lang should be a two-letter language code');

        return this._form.compile(lang).then((compiled) => {
            let csrf = '';
            let scopes = null;

            compiled.forEach((field) => {
                if (field.id === 'csrf') {
                    csrf = field.value;
                    return;
                }

                if (field.id === 'scopes' && !scopes) {
                    scopes = field.value;
                }
            });

            this._store.editPage.allScopes = scopes;
            this._store.editPage.csrf = csrf;

            /* eslint-disable global-require */
            if (process.env.NODE_ENV === 'development') {
                reactTemplates[lang] = require(`../../../templates/${reactPage}.react.${lang}.js`).default;
            }
            /* eslint-enable global-require */

            const getRootComponent = reactTemplates[lang];
            const reactMarkup = getRootComponent(this._store, this._controller.getTld(), lang);

            return {
                editClient: compiled,
                clientId: this._clientId,
                reactMarkup,
                reduxState: JSON.stringify(this._store)
            };
        });
    }
});
