const i18n = require('putils').i18n;
const inherit = require('inherit');
const Pview = require('pview');
const assert = require('assert');
const OAuthApi = require('papi/OAuth');
const ClientModel = require('papi/OAuth/models/Client');
const ScopesList = require('../../../blocks/clientScopesList/clientScopesView');

function getTimes(lang) {
    return {
        year: {
            one: i18n(lang, 'time.year.one'),
            some: i18n(lang, 'time.year.some'),
            many: i18n(lang, 'time.year.many')
        },
        day: {
            one: i18n(lang, 'time.day.one'),
            some: i18n(lang, 'time.day.some'),
            many: i18n(lang, 'time.day.many')
        },
        hour: {
            one: i18n(lang, 'time.hour.one'),
            some: i18n(lang, 'time.hour.some'),
            many: i18n(lang, 'time.hour.many')
        },
        minute: {
            one: i18n(lang, 'time.minute.one'),
            some: i18n(lang, 'time.minute.some'),
            many: i18n(lang, 'time.minute.many')
        },
        second: {
            one: i18n(lang, 'time.second.one'),
            some: i18n(lang, 'time.second.some'),
            many: i18n(lang, 'time.second.many')
        }
    };
}

function getPluralForm(amount) {
    const num = Math.round(amount);
    const numStr = String(num);
    const lastDigit = numStr[numStr.length - 1];
    const some = ['2', '3', '4'];
    const many = ['0', '5', '6', '7', '8', '9'];

    if (num >= 11 && num <= 19) {
        return 'many';
    } else if (some.includes(lastDigit)) {
        return 'some';
    } else if (many.includes(lastDigit)) {
        return 'many';
    }

    return 'one';
}

function formatTokenLifeSpan(ttl, lang) {
    let amount;
    let message;
    const YEAR = 31536000;
    const DAY = 86400;
    const HOUR = 3600;
    const MINUTE = 60;
    const times = getTimes(lang);

    if (ttl === null || ttl === 0) {
        return i18n(lang, 'app.ttl.infinite');
    } else if (ttl >= YEAR) {
        amount = ttl / YEAR;
        message = times.year[getPluralForm(amount)];
    } else if (ttl >= DAY) {
        amount = ttl / DAY;
        message = times.day[getPluralForm(amount)];
    } else if (ttl >= HOUR) {
        amount = ttl / HOUR;
        message = times.hour[getPluralForm(amount)];
    } else if (ttl >= MINUTE) {
        amount = ttl / MINUTE;
        message = times.minute[getPluralForm(amount)];
    } else {
        amount = ttl;
        message = times.second[getPluralForm(ttl)];
    }

    return `${amount} ${message}`;
}

function isTokenRefreshable(scopes) {
    if (!scopes) {
        return false;
    }

    return scopes.every((scope) => scope.is_ttl_refreshable || !scope.ttl || scope.ttl === 'INFINITE');
}

function formatDate(time) {
    const date = (time.getDate() < 10 ? '0' : '') + time.getDate();
    const month = (time.getMonth() < 9 ? '0' : '') + (time.getMonth() + 1);
    const year = time.getFullYear();
    return `${date}.${month}.${year}`;
}

module.exports = inherit(Pview, {
    name: 'client.info',

    __constructor: function (api, clientId) {
        assert(api instanceof OAuthApi, 'Api instance expected');
        assert(ClientModel.isValidClientId(clientId), 'Client id should be valid');

        this._api = api;
        this._clientId = clientId;
        this.__base.apply(this, arguments);
    },

    _compile: function (lang) {
        const clientID = this._clientId;
        return this._api.clientInfoVer3(clientID).then(function (client) {
            return new ScopesList(client.getScopes()).compile().then(function (scopes) {
                const clientInfo = {
                    viewedByOwner: false,
                    title: client.getTitle(),
                    description: client.getDescription(),
                    homepage: client.getHomepage(),
                    scopes: scopes.clientScopesList,
                    icon: client.getIcon(),
                    creation_time: formatDate(client.getCreationTime()),
                    isYandex: client.isYandex()
                };

                if (client.isViewedByOwner()) {
                    const ttl = client.getJSONResponse().token_ttl;
                    const formatedToken = formatTokenLifeSpan(ttl, lang);

                    Object.assign(clientInfo, {
                        viewedByOwner: true,
                        clientId: clientID,
                        secret: client.getSecret(),
                        blocked: client.isBlocked(),
                        editable: client.isEditable(),
                        approval_status: client.getApprovalStatus(),
                        callback_url: client.getCallbackUrl(),
                        token_ttl:
                            isTokenRefreshable(client.getScopes()) && !(ttl === null || ttl === 0)
                                ? `${i18n(lang, 'app.ttl.at_least')} ${formatedToken}`
                                : formatedToken
                    });
                }

                return {client: clientInfo};
            });
        });
    }
});
