const _ = require('lodash');
const assert = require('assert');
const PView = require('pview');
const inherit = require('inherit');
const config = require('../../../config/current');
const {mapClient, mapClientsList} = require('./utils');

const HAS_PLUS_ATTRIBUTE = 1015;

const reactTemplates = {};

const createReactTemplate = (lang) => {
    reactTemplates[lang] = require(`../../../templates/client.new.react.${lang}.js`).default;
};

if (process.env.NODE_ENV !== 'development') {
    config.loc.langs.forEach(createReactTemplate);
}

module.exports = inherit(PView, {
    name: 'client.formView',
    __constructor: function (controller, csrf, uatraits, userInfo, sessionId, client, clientList) {
        this.__base.apply(this, arguments);

        this._controller = controller;
        this._csrf = csrf;
        this._uatraits = uatraits;
        this._userInfo = userInfo;
        this._client = client;
        this._sessionId = sessionId;
        this._clientList = clientList;
    },

    _compile: async function (lang) {
        assert(typeof lang === 'string' && lang.length === 2, 'Lang should be a two-letter language code');

        const allScopes = _.sortBy(
            this._userInfo.scopes._scopes.reduce((acc, scope) => {
                const newPermission = {
                    id: scope._id,
                    title: scope._title,
                    turnedOn:
                        this._client.client && this._client.client.id
                            ? Object.values(this._client.client.scopes).some((i) => i[scope._id])
                            : false,
                    ttl: scope._ttl,
                    section: scope._section,
                    tags: scope._tags,
                    ttlRefreshable: scope._ttlRefreshable,
                    requiresApproval: scope._requiresApproval
                };
                const section = acc.find((i) => i.section === scope._section);

                if (section) {
                    section.permissions.push(newPermission);
                } else {
                    acc.push({
                        section: scope._section,
                        id: `ScopesSection${acc.length}`,
                        permissions: [newPermission]
                    });
                }

                return acc;
            }, []),
            'section'
        );

        const {
            connection_id: ci,
            attributes = {},
            login = '',
            display_name: {avatar: {default: avatarId = '0/0-0'} = {}, name: displayName = ''} = {}
        } = this._sessionId;

        const {permissionsById, sections} = allScopes.reduce(
            (acc, scope) => {
                scope.permissions.forEach((permission) => {
                    acc.permissionsById[permission.id] = permission;
                });
                acc.sections.push({
                    title: scope.section,
                    permissions: scope.permissions.map((permission) => permission.id)
                });
                return acc;
            },
            {permissionsById: {}, sections: []}
        );

        this._client.client.token_ttl = this._client.token_ttl;

        const curClient = mapClient(this._client.client);

        const store = {
            edit: curClient,
            clients: mapClientsList(this._clientList),
            common: {
                retpath: this._controller.getUrl().href,
                isCorporate: this._userInfo.allowRegisteringYandexClients,
                userInfo: this._userInfo,
                permissionsById,
                sections,
                csrf: this._csrf,
                documentationLinks: config.documentationLinks
            },
            settings: {
                tld: this._controller.getTld(),
                lang,
                host: this._controller.getUrl().hostname,
                year: new Date().getFullYear(),
                devUrl: `https://${this._controller.getUrl().hostname}/verification_code`,
                passportHost: this._controller.getPassportHost(),
                isTesting: ['development', 'testing'].includes(process.env.NODE_ENV)
            },
            uatraits: this._uatraits,
            userInfo: {
                ci,
                login,
                avatarId,
                displayName,
                avatarPath: (config.mds || '')
                    .replace('/islands-200', '/islands-retina-middle')
                    .replace('%avatarId%', avatarId),
                hasPlus: Boolean(attributes[HAS_PLUS_ATTRIBUTE])
            },

            metrics: {
                header: 'oauth/client',
                experiments: this._controller._response.locals.experiments.encodedBoxes || ''
            }
        };

        if (!this._clientList.length && this._client.client.id) {
            store.clients.idList.push(curClient.clientId);
            store.clients.clientsMap[curClient.clientId] = curClient;
        }

        if (process.env.NODE_ENV === 'development') {
            createReactTemplate(lang);
        }

        return {
            reactMarkup: reactTemplates[lang](store, this._controller.getUrl().pathname),
            reduxState: JSON.stringify(store)
        };
    }
});
