'use strict';

const when = require('when');
const OAuth = require('papi/OAuth');
const Layout = require('./DeviceLayout');
const ApiError = OAuth.ApiError;
const ReactView = require('./ReactView');
const inherit = require('inherit');
const AbstractPage = require('../AbstractPage');

module.exports = inherit(AbstractPage, {
    name: 'device',

    open: function () {
        const controller = this._controller;
        const logger = this._logger;
        const origin = controller.getRequestParam('origin');

        if (!controller.getAuth().isLoggedIn()) {
            const tld = controller.getTld();
            const queryParams = {
                origin: origin || 'passport_device',
                noreturn: 1,
                backpath: 'https://yandex.%tld%'.replace('%tld%', tld)
            };

            logger.info('User not logged in, redirecting to auth');
            controller.getAuth().authorize(queryParams);
            return when.reject();
        }

        if (controller.getAuth().isAutologged()) {
            logger.info('User logged in with autologin, redirecting to lightauth2full');
            controller.getAuth().lightauth2full(true);
            return when.reject();
        }

        logger.info('opened');

        const method = controller.getMethod();

        if (method.isPost()) {
            return this._submit();
        }

        return this._render();
    },

    _submit: function () {
        const url = this._controller.getUrl().pathname.replace(/\/+$/, '');
        const params = url.match(/^\/device(\/(allow|deny))?$/) || [];

        if (params && params[2]) {
            const action = params[2];

            switch (action) {
                case 'allow':
                    return this._allow();
                case 'deny':
                    return this._deny();
            }
        } else {
            return this._sendCode();
        }

        return this._render();
    },

    _sendCode: async function () {
        const self = this;
        const logger = this._logger;
        const controller = this._controller;
        const deviceCode = controller.getRequestParam('code');
        const clientId = controller.getRequestParam('client_id');

        let csrfValid;

        try {
            csrfValid = await this._controller.isCsrfTokenValidV2();
        } catch {
            csrfValid = false;
        }

        if (!csrfValid) {
            logger.error('CSRF-token is not valid');
            return controller._response.json({
                status: 'error',
                errors: ['form.internal']
            });
        }

        return self._api
            .deviceAuthorizeSubmit(deviceCode, clientId)
            .then(function (response) {
                controller._response.json(response);
            })
            .catch(function (error) {
                if (error instanceof ApiError) {
                    controller._response.json(error._response);
                } else {
                    logger.info('Not an API error, gets rethrown: %s', error);
                    throw error;
                }
            });
    },

    _allow: async function () {
        const self = this;
        const logger = this._logger;
        const controller = this._controller;
        const deviceCode = controller.getRequestParam('code');
        const clientId = controller.getRequestParam('client_id');

        let csrfValid;

        try {
            csrfValid = await controller.isCsrfTokenValidV2();
        } catch {
            csrfValid = false;
        }

        if (!csrfValid) {
            logger.error('CSRF-token is not valid');
            return controller._response.json({
                status: 'error',
                errors: ['form.internal']
            });
        }

        return self._api
            .deviceAuthorizeCommit(deviceCode, clientId)
            .then(function (response) {
                controller._response.json(response);
            })
            .catch(function (error) {
                if (error instanceof ApiError) {
                    controller._response.json(error._response);
                } else {
                    logger.info('Not an API error, gets rethrown: %s', error);
                    throw error;
                }
            });
    },

    _deny: function () {
        return this._render();
    },

    _render: function () {
        const controller = this._controller;
        const logger = this._logger;

        logger.debug('Rendering');
        return new Layout(controller)
            .append(new ReactView(controller))
            .render(controller.getLang())
            .then(function (rendered) {
                logger.verbose('Sending', rendered);
                return controller.sendPage(rendered);
            })
            .then(function () {
                logger.info('Page sent');
            });
    }
});
