const _ = require('lodash');
const url = require('url');
const assert = require('assert');
const inherit = require('inherit');
const when = require('when');
const PView = require('pview');
const serialize = require('serialize-javascript');
const config = require('../../config/current');
const {SERVICE_ALIASES} = require('../../lib/tvm');
const langs = config.loc.langs;
const reactTemplates = {};
const reactPage = 'device';

/* eslint-disable global-require */
if (process.env.NODE_ENV !== 'development') {
    langs.forEach(function (lang) {
        reactTemplates[lang] = require(`../../templates/${reactPage}.react.${lang}.js`).default;
    });
}

module.exports = inherit(PView, {
    name: 'device.reactView',

    __constructor: function (controller) {
        this.__base.apply(this, arguments);

        this._controller = controller;
        this._reactPage = 'device';
        this._store = {
            devicePage: {
                csrf: '',
                origin: controller.getRequestParam('origin') || null,
                deviceCode: '',
                currentUser: {},
                isFetching: false,
                errors: [],
                clientScopes: [],
                clientTitle: '',
                clientId: controller.getRequestParam('client_id') || null,
                redirectUrl:
                    (controller.getRequestParam('retpath') &&
                        url.format(
                            Object.assign({}, controller.getAuthUrl(), {
                                pathname: 'redirect',
                                query: {
                                    url: controller.getRequestParam('retpath'),
                                    origin: controller.getRequestParam('origin') || 'oauth'
                                }
                            })
                        )) ||
                    null,
                status: ''
            },
            settings: {
                tld: controller.getTld(),
                language: controller.getLang(),
                passportHost: controller.getPassportHost()
            }
        };
    },

    _compile: function (lang) {
        assert(typeof lang === 'string' && lang.length === 2, 'Lang should be a two-letter language code');
        const self = this;
        const controller = this._controller;
        const req = controller._request;
        const headers = {
            'X-Ya-Service-Ticket': req.serviceTickets && req.serviceTickets[SERVICE_ALIASES.BLACKBOX]
        };

        return when
            .all([controller.getAuth().sessionID({}, headers), controller.getCsrfToken()])
            .then(function (responses) {
                const auth = responses[0];
                const csrf = responses[1];

                if (!auth || !csrf) {
                    return {};
                }

                self._setCurrentUser(auth);
                self._store.devicePage.csrf = csrf;

                /* eslint-disable global-require */
                if (process.env.NODE_ENV === 'development') {
                    reactTemplates[lang] = require(`../../templates/${reactPage}.react.${lang}.js`).default;
                }
                /* eslint-enable global-require */

                const getRootComponent = reactTemplates[lang];
                const reactMarkup = getRootComponent(self._store);

                return {
                    reactMarkup,
                    reduxState: serialize(self._store, {isJSON: true})
                };
            });
    },

    _setCurrentUser: function (user) {
        const controller = this._controller;
        const displayName = user.display_name || {};
        const avatarId = displayName.avatar && displayName.avatar.default;
        let avatarPath = _.clone(config.mds) || '';
        avatarPath = avatarPath.replace('%avatarId%', avatarId);

        this._store.devicePage.currentUser = {
            displayName: displayName.name,
            avatarPath: avatarPath,
            login: user.login,
            changeAccountLink: url.format(
                _.extend(controller.getAuthUrl(), {
                    query: {
                        retpath: url.format(controller.getUrl()),
                        origin: controller.getRequestParam('origin') || 'oauth'
                    }
                })
            )
        };
    }
});
