const inherit = require('inherit');
const assert = require('assert');
const AbstractPage = require('../AbstractPage');
const ErrorView = require('./ErrorView.js');
const ErrorLayout = require('./ErrorLayout.js');
const HTTPException = require('../../lib/exceptions/http/HTTPException');

module.exports = inherit(AbstractPage, {
    name: 'error',

    __constructor: function () {
        this.__base.apply(this, arguments);
        this._planned = false;
    },

    open: function () {
        const error = this._error;
        assert(error instanceof Error, 'Error should be defined via ErrorPage.setError() method');
        this._logger.info('opened for %s', error);

        const code =
            error instanceof HTTPException
                ? error.code //HTTP Errors are sent with their codes
                : 500; //All others codes are 500 internal errors

        this._logger.debug('Rendering the layout for', error);
        return this._getLayout()
            .append(new ErrorView(error, this._planned))
            .render()
            .then((rendered) => {
                this._logger.verbose('Sending %s with code %s', rendered, code);
                return this._controller.sendPage(rendered, code);
            })
            .then(() => this._logger.info('Page sent with code %s', code));
    },

    _getLayout: function () {
        return new ErrorLayout(this._controller);
    },

    /**
     * Set an error to open the page with
     * @param {Error|*} error   Anything that is not an error instance gets converted to one
     * @returns {ErrorPage}
     */
    setError: function (error) {
        if (!error instanceof Error) {
            error = new Error(error);
        }

        this._error = error;

        if (error instanceof HTTPException) {
            this.setPlanned(); //HTTP Errors are always planned, since they are raised by the app
        }

        return this;
    },

    /**
     * Set this error planned, so that it gets shown in production
     * Otherwise the generic error is shown
     * @returns {ErrorPage}
     */
    setPlanned: function () {
        this._planned = true;
        return this;
    }
});
