const express = require('express');
const uuid = require('node-uuid');
const cookieParser = require('cookie-parser');
const bodyParser = require('body-parser');
const formidable = require('express-formidable');
const helmet = require('helmet');
const app = express();
const config = require('./config/current');
const psl = require('psl');
const lookup = require('./lib/getLookup.js');

const WEBVISOR_AND_METRICKA_DOMAINS =
    // eslint-disable-next-line no-useless-escape, max-len
    /^([^\/]+\.)?(oauth.yandex\.(?:ru|ua|by|kz|com|com\.tr)|webvisor\.com|metri[ck]a\.yandex\.(com|ru|by|com\.tr))$/;

app.disable('trust proxy')
    .use(cookieParser())
    .use(
        bodyParser.urlencoded({
            extended: false
        })
    )
    .use((req, res, next) => {
        res.locals.nonce = uuid.v4();
        next();
    })
    .use((req, res, next) => {
        const {tld = ''} = psl.parse(req.hostname);

        res.locals.selfTld = tld === 'am' ? 'com.am' : tld || 'ru';
        next();
    })
    .use((err, req, res, next) => {
        res.status(err.status);
        res.json({
            status: 'error',
            message: err.message
        });
    })
    .use(formidable.parse());

app.use((req, res, next) => {
    const iP = req.get('x-real-ip');

    if (iP) {
        const net = lookup.getIpTraits(iP);

        res.locals.isYandexInternalNetwork = net && Boolean(net.yandex_net);
        res.locals.isYandex = net && Boolean(net.yandex_net) && Boolean(net.yandex_staff);
    }

    return next();
});

app.use(helmet.contentSecurityPolicy(config.helmet.csp));
app.use((req, res, next) => {
    if (res.locals.isYandex && WEBVISOR_AND_METRICKA_DOMAINS.test(req.hostname)) {
        return next();
    }
    return helmet.frameguard(config.helmet.frameguard)(req, res, next);
});
app.use(helmet.hsts());
app.use(helmet.hidePoweredBy());
app.use(helmet.ieNoOpen());
app.use(helmet.noSniff());
app.use(helmet.noCache(config.helmet.noCache));
app.use(helmet.dnsPrefetchControl());
app.use(helmet.xssFilter());

if (app.get('env') === 'development') {
    app.use('/st', express.static(__dirname + '/public'));
}

module.exports = app;
