import u from 'updeep';
import Api from '../api';
import metrics from '../../../metrics';

const METRIC_CREATE_APP_FORM_ERROR = 'METRIC_CREATE_APP_FORM_ERROR';

export const EDIT_PAGE_UPDATE_TEXT_FIELD = 'EDIT_PAGE_UPDATE_TEXT_FIELD';
export const EDIT_PAGE_UPDATE_SECTION = 'EDIT_PAGE_UPDATE_SECTION';
export const EDIT_PAGE_ADD_ITEM = 'EDIT_PAGE_ADD_ITEM';
export const EDIT_PAGE_DELETE_ITEM = 'EDIT_PAGE_DELETE_ITEM';
export const EDIT_PAGE_TOGGLE_SCOPE_PERMISSIONS = 'EDIT_PAGE_TOGGLE_SCOPE_PERMISSIONS';
export const EDIT_PAGE_SELECT_PERMISSION = 'EDIT_PAGE_SELECT_PERMISSION';
export const EDIT_PAGE_SELECT_ICON = 'EDIT_PAGE_SELECT_ICON';
export const EDIT_PAGE_DELETE_ICON = 'EDIT_PAGE_DELETE_ICON';
export const EDIT_PAGE_SHOW_WARNING = 'EDIT_PAGE_SHOW_WARNING';
export const EDIT_PAGE_SET_FIELD_ERROR = 'EDIT_PAGE_SET_FIELD_ERROR';
export const EDIT_PAGE_REMOVE_ERROR = 'EDIT_PAGE_REMOVE_ERROR';

export const EDIT_PAGE_SAVE_APPLICATION = 'EDIT_PAGE_SAVE_APPLICATION';
export const EDIT_PAGE_SAVE_APPLICATION_SUCCESS = 'EDIT_PAGE_SAVE_APPLICATION_SUCCESS';
export const EDIT_PAGE_SAVE_APPLICATION_FAIL = 'EDIT_PAGE_SAVE_APPLICATION_FAIL';

export const EDIT_PAGE_VALIDATE_APPLICATION = 'EDIT_PAGE_VALIDATE_APPLICATION';
export const EDIT_PAGE_VALIDATE_APPLICATION_SUCCESS = 'EDIT_PAGE_VALIDATE_APPLICATION_SUCCESS';
export const EDIT_PAGE_VALIDATE_APPLICATION_FAIL = 'EDIT_PAGE_VALIDATE_APPLICATION_FAIL';

export function updateTextField(name, value) {
    return {
        type: EDIT_PAGE_UPDATE_TEXT_FIELD,
        name,
        value
    };
}

export function showSection(section) {
    return {
        type: EDIT_PAGE_UPDATE_SECTION,
        section
    };
}

export function deleteItem(name, index) {
    return {
        type: EDIT_PAGE_DELETE_ITEM,
        name,
        index
    };
}

export function addItem(name) {
    return {
        type: EDIT_PAGE_ADD_ITEM,
        name
    };
}

export function toggleScopePermissions(sectionId) {
    return {
        type: EDIT_PAGE_TOGGLE_SCOPE_PERMISSIONS,
        sectionId
    };
}

export function selectPermission(checked, section, permission) {
    return {
        type: EDIT_PAGE_SELECT_PERMISSION,
        checked,
        section,
        permission
    };
}

export function selectIcon(file) {
    return {
        type: EDIT_PAGE_SELECT_ICON,
        file
    };
}

export function deleteIcon() {
    return {
        type: EDIT_PAGE_DELETE_ICON
    };
}

export function showWarning(show) {
    return {
        type: EDIT_PAGE_SHOW_WARNING,
        show
    };
}

export function setFieldError(fieldName, error, index) {
    return {
        type: EDIT_PAGE_SET_FIELD_ERROR,
        fieldName,
        error,
        index
    };
}

export function validateApplicationChanges(fieldName) {
    return function(dispatch, getState) {
        const {editPage: {form, clientId, csrf, errors: stateErrors}} = getState();
        const formData = new FormData();
        const data = u({
            icon_file: undefined,
            scopes: u.constant(form.scopes.map((scope) => scope.name))
        }, form);

        formData.append('data', JSON.stringify(data));
        formData.append('csrf', csrf);

        dispatch({
            type: EDIT_PAGE_VALIDATE_APPLICATION
        });

        return Api.validateApplicationChanges(clientId, formData)
            .then(function(res) {
                dispatch({
                    type: EDIT_PAGE_VALIDATE_APPLICATION_SUCCESS,
                    invalidateTokens: res.invalidate_tokens,
                    tokenLifeSpan: res.token_ttl,
                    shouldBeModerated: res.will_require_approval
                });
            })
            .catch(function(res) {
                if (fieldName) {
                    let errors = res.errors;


                    if (errors.length) {
                        const error = errors.find(error => error.indexOf(fieldName) === 0);

                        dispatch(setFieldError(error || errors[0]));
                    }

                    if (stateErrors && Object.keys(stateErrors).length) {
                        Object.keys(stateErrors).forEach(errorKey => {
                            if (errors.every(error => error.indexOf(errorKey) !== 0)) {
                                dispatch({
                                    type: EDIT_PAGE_REMOVE_ERROR,
                                    errorKey
                                });
                            }
                        })
                    }
                }
            });
    };
}

export function saveApplication(validate) {
    return function(dispatch, getState) {
        const {editPage: {form, clientId, csrf}} = getState();
        const formData = new FormData();
        let validateApplication = Promise.resolve();

        if (form.icon_file) {
            formData.append('icon', form.icon_file);
        }

        const data = u({
            icon_file: undefined,
            scopes: u.constant(form.scopes.map((scope) => scope.name))
        }, form);

        formData.append('data', JSON.stringify(data));
        formData.append('csrf', csrf);

        dispatch({
            type: EDIT_PAGE_SAVE_APPLICATION
        });

        if (validate) {
            validateApplication = Api.validateApplicationChanges(clientId, formData)
                .then(function(res) {
                    dispatch({
                        type: EDIT_PAGE_VALIDATE_APPLICATION_SUCCESS,
                        invalidateTokens: res.invalidate_tokens,
                        tokenLifeSpan: res.token_ttl,
                        shouldBeModerated: res.will_require_approval,
                        showWarningIfNeeded: true
                    });

                    if (res.invalidate_tokens) {
                        return Promise.reject({});
                    }

                    return Promise.resolve();
                });
        }

        return validateApplication
            .then(function() {
                return Api.saveApplication(clientId, formData);
            })
            .then(function(res) {
                const cid = res.client_id || clientId;

                dispatch({
                    type: EDIT_PAGE_SAVE_APPLICATION_SUCCESS
                });
                location.replace(`/client/${cid}`);
            })
            .catch(function(res) {
                if (!clientId) {
                    metrics.send([METRIC_CREATE_APP_FORM_ERROR, res.errors]);
                }
                dispatch({
                    type: EDIT_PAGE_SAVE_APPLICATION_FAIL,
                    errors: res.errors
                });
            });
    };
}
