import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import classnames from 'classnames';
import TextInput from 'lego-on-react/src/components/textinput/textinput.react';
import TextArea from 'lego-on-react/src/components/textarea/textarea.react';
import Button from 'lego-on-react/src/components/button/button.react';
import Link from 'lego-on-react/src/components/link/link.react';
import Modal from 'lego-on-react/src/components/modal/modal.react';
import Spinner from 'lego-on-react/src/components/spin/spin.react';
import Checkbox from 'lego-on-react/src/components/checkbox/checkbox.react';
import metrics from '../../../metrics';
import './Edit.styl';

import {
    updateTextField,
    showSection,
    deleteItem,
    addItem,
    toggleScopePermissions,
    selectPermission,
    selectIcon,
    deleteIcon,
    saveApplication,
    showWarning,
    setFieldError,
    validateApplicationChanges
} from '../actions/edit';


const errorMessages = {
    title: {
        missing: i18n('frontend.fields.field._common_.error.missingvalue'),
        too_long: i18n('frontend.fields.field.title.error.toolong')
    },
    description: {
        too_long: i18n('frontend.fields.field.description.error.toolong'),
        missing: i18n('frontend.fields.field.description.error.missing')
    },
    homepage: {
        invalid: i18n('frontend.fields.field.url.error.invalid'),
        scheme_forbidden: i18n('frontend.fields.field.callback.error.scheme_forbidden'),
        too_long: i18n('frontend.fields.field._common_.error.url_too_long'),
        not_absolute: i18n('frontend.fields.field.url.error.invalid'),
        required: i18n('frontend.fields.field._common_.error.missingvalue'),
        scheme_missing: i18n('frontend.fields.field.url.error.invalid')
    },
    redirect_uri: {
        required: i18n('frontend.fields.field._common_.error.missingvalue'),
        missing: i18n('frontend.fields.field._common_.error.missingvalue'),
        invalid: i18n('frontend.fields.field.url.error.invalid'),
        too_long: i18n('frontend.fields.field._common_.error.url_too_long'),
        scheme_missing: i18n('frontend.fields.field.url.error.invalid'),
        scheme_forbidden: i18n('frontend.fields.field.callback.error.scheme_forbidden'),
        not_absolute: i18n('frontend.fields.field.url.error.invalid')
    },
    icon: {
        bad_format: i18n('frontend.fields.field.icon.error.badformat'),
        required: i18n('frontend.fields.field._common_.error.missingvalue')
    },
    icon_file: {
        invalid_format: i18n('frontend.fields.field.icon.error.badformat'),
        too_large: i18n('frontend.fields.field.icon.error.toolarge')
    },
    ios_app_id: {
        too_many: i18n('frontend.authorize.authorize.errors.too_many.ios'),
        required: i18n('frontend.fields.field._common_.error.missingvalue'),
        missing: i18n('frontend.fields.field._common_.error.missingvalue'),
        invalid: i18n('frontend.fields.field._common_.error.invalid')
    },
    ios_appstore_url: {
        invalid: i18n('frontend.fields.field.url.error.invalid'),
        scheme_forbidden: i18n('frontend.fields.field.callback.error.scheme_forbidden'),
        too_long: i18n('frontend.fields.field._common_.error.url_too_long'),
        not_absolute: i18n('frontend.fields.field.url.error.invalid'),
        required: i18n('frontend.fields.field._common_.error.missingvalue'),
        scheme_missing: i18n('frontend.fields.field.url.error.invalid')
    },
    android_appstore_url: {
        invalid: i18n('frontend.fields.field.url.error.invalid'),
        scheme_forbidden: i18n('frontend.fields.field.callback.error.scheme_forbidden'),
        too_long: i18n('frontend.fields.field._common_.error.url_too_long'),
        not_absolute: i18n('frontend.fields.field.url.error.invalid'),
        required: i18n('frontend.fields.field._common_.error.missingvalue'),
        scheme_missing: i18n('frontend.fields.field.url.error.invalid')
    },
    android_package_name: {
        too_many: i18n('frontend.authorize.authorize.errors.too_many.android'),
        required: i18n('frontend.fields.field._common_.error.missingvalue'),
        missing: i18n('frontend.fields.field._common_.error.missingvalue'),
        invalid: i18n('frontend.fields.field._common_.error.invalid')
    },
    android_cert_fingerprints: {
        required: i18n('frontend.fields.field._common_.error.missingvalue'),
        invalid: i18n('frontend.fields.field._common_.error.invalid'),
        missing: i18n('frontend.fields.field._common_.error.missingvalue')
    },
    scopes: {
        missing: i18n('frontend.fields.field.scopes.error.missingvalue')
    },
    form: {
        internal: i18n('frontend.authorize.authorize.errors.try_later')
    },
    turboapp_base_url: {
        required: i18n('frontend.fields.field._common_.error.missingvalue'),
        scheme_missing: i18n('frontend.fields.field.url.error.invalid'),
        scheme_forbidden: i18n('frontend.fields.field.callback.error.scheme_forbidden'),
        not_absolute: i18n('frontend.fields.field.url.error.invalid')
    }
};

const times = {
    year: {
        one: i18n('frontend.time.time.year.one'),
        some: i18n('frontend.time.time.year.some'),
        many: i18n('frontend.time.time.year.many')
    },
    day: {
        one: i18n('frontend.time.time.day.one'),
        some: i18n('frontend.time.time.day.some'),
        many: i18n('frontend.time.time.day.many')
    },
    hour: {
        one: i18n('frontend.time.time.hour.one'),
        some: i18n('frontend.time.time.hour.some'),
        many: i18n('frontend.time.time.hour.many')
    },
    minute: {
        one: i18n('frontend.time.time.minute.one'),
        some: i18n('frontend.time.time.minute.some'),
        many: i18n('frontend.time.time.minute.many')
    },
    second: {
        one: i18n('frontend.time.time.second.one'),
        some: i18n('frontend.time.time.second.some'),
        many: i18n('frontend.time.time.second.many')
    }
};

const OAUTH_SUPPORT_LINK = 'https://yandex.%tld%/dev/oauth/doc/dg/tasks/register-client.html/';
const SUPPORT_DOMAINS = ['ru', 'ua', 'by', 'kz'];
const METRIC_APP_NEW_PAGE_OPEN = 'METRIC_APP_NEW_PAGE_OPEN';

function getPluralForm(amount) {
    const num = Math.round(amount);
    const numStr = String(num);
    const lastDigit = numStr[numStr.length - 1];
    const some = ['2', '3', '4'];
    const many = ['0', '5', '6', '7', '8', '9'];

    if (num >= 11 && num <= 19) {
        return 'many';
    } else if (some.includes(lastDigit)) {
        return 'some';
    } else if (many.includes(lastDigit)) {
        return 'many';
    }

    return 'one';
}

class EditPage extends React.PureComponent {
    static mapStateToProps(state) {
        return state.editPage;
    }

    static formatTokenLifeSpan(ttl) {
        let amount;
        let message;
        const YEAR = 31536000;
        const DAY = 86400;
        const HOUR = 3600;
        const MINUTE = 60;

        if (ttl === null || ttl === 0) {
            return i18n('frontend.app.app.ttl.infinite');
        } else if (ttl >= YEAR) {
            amount = ttl / YEAR;
            message = times.year[getPluralForm(amount)];
        } else if (ttl >= DAY) {
            amount = ttl / DAY;
            message = times.day[getPluralForm(amount)];
        } else if (ttl >= HOUR) {
            amount = ttl / HOUR;
            message = times.hour[getPluralForm(amount)];
        } else if (ttl >= MINUTE) {
            amount = ttl / MINUTE;
            message = times.minute[getPluralForm(amount)];
        } else {
            amount = ttl;
            message = times.second[getPluralForm(ttl)];
        }

        return `${amount} ${message}`;
    }

    constructor(props) {
        super(props);

        const tld = this.props.tld;

        this.language = this.props.lang;
        this.domain = SUPPORT_DOMAINS.find((el) => el === tld) ? 'ru' : 'com';
        this.onTextFieldChange = this.onTextFieldChange.bind(this);
        this.onYandexAppCheckboxChange = this.onYandexAppCheckboxChange.bind(this);
        this.onSectionCheck = this.onSectionCheck.bind(this);
        this.onAddButtonClick = this.onAddButtonClick.bind(this);
        this.onDeleteIconClick = this.onDeleteIconClick.bind(this);
        this.onFileUpload = this.onFileUpload.bind(this);
        this.onSaveButtonClick = this.onSaveButtonClick.bind(this);
        this.onFormSubmit = this.onFormSubmit.bind(this);
        this.hideWarning = this.hideWarning.bind(this);
        this.onFieldBlur = this.onFieldBlur.bind(this);
        this.onTextAreaFocus = this.onTextAreaFocus.bind(this);
        this.onFieldFocus = this.onFieldFocus.bind(this);
        this.setURLForDevelopment = this.setURLForDevelopment.bind(this);
    }

    componentDidMount() {
        const { clientId } = this.props;
        if (!clientId) {
            metrics.send(METRIC_APP_NEW_PAGE_OPEN)
        }
    }

    setURLForDevelopment(index) {
        const dispatch = this.props.dispatch;
        const link = document.createElement('a');

        link.href = window.location.href;
        link.pathname = 'verification_code';

        dispatch(setFieldError('redirect_uri', null, index));
        dispatch(updateTextField(`redirect_uri.${index}`, link.href));
    }

    onFieldBlur(e) {
        const name = e.target.name;

        this.props.dispatch(validateApplicationChanges(name));
    }

    onFieldFocus(event) {
        const parts = event.target.name.split('.');

        this.props.dispatch(setFieldError(parts[0], null, parts[1]));
    }

    onTextAreaFocus(focus, props) {
        const dispatch = this.props.dispatch;
        const name = props.name;

        if (focus) {
            dispatch(setFieldError(name, null));
        } else {
            dispatch(validateApplicationChanges(name));
        }
    }

    hideWarning() {
        this.props.dispatch(showWarning(false));
    }

    onTextFieldChange(value, props) {
        const name = props.name;

        this.props.dispatch(updateTextField(name, value));
    }

    onYandexAppCheckboxChange(event) {
        this.props.dispatch(updateTextField('is_yandex', event.target.checked));
    }

    onSectionCheck(event) {
        const section = event.target.name;

        this.props.dispatch(showSection(section));
    }

    onSaveButtonClick(event) {
        event.preventDefault();
        this.props.dispatch(saveApplication());
    }

    onFormSubmit(event) {
        event.preventDefault();
        const formSubmitPromise = this.props.dispatch(saveApplication(true));

        formSubmitPromise.then(() => {
            const {errors} = this.props;

            if (Object.keys(errors).length === 0) {
                return;
            }

            const error = this.refs.form.querySelector('.js-error');
            const currentScroll = document.documentElement.scrollTop || document.body.scrollTop;
            const errorTop = error.getBoundingClientRect().top - 100;
            const animationLength = 300;
            let start = null;

            function animateScrolling(timestamp) {
                if (!start) {
                    start = timestamp;
                }
                const progress = timestamp - start;
                scrollTo(0, currentScroll + errorTop * (progress / animationLength));
                if (progress < animationLength) {
                    requestAnimationFrame(animateScrolling);
                }
            }

            requestAnimationFrame(animateScrolling);
        });
    }

    onFileUpload(event) {
        const dispatch = this.props.dispatch;
        const file = event.target.files[0];

        event.preventDefault();

        if (file) {
            dispatch(selectIcon(file));
        }
    }

    onDeleteIconClick() {
        const dispatch = this.props.dispatch;

        this.refs.icon.clear();
        dispatch(deleteIcon());
    }

    onAddButtonClick(event) {
        const dispatch = this.props.dispatch;
        const name = event.target.name;

        event.preventDefault();
        dispatch(addItem(name));
    }

    renderError(fieldName, fieldIndex) {
        const errors = this.props.errors;
        let code;
        let errorMessage;

        if (Object.keys(errors).length === 0) {
            return null;
        }

        if (fieldIndex !== undefined) {
            code = errors[fieldName] && errors[fieldName][fieldIndex];
        } else {
            code = errors[fieldName];
        }

        if (code) {
            errorMessage = errorMessages[fieldName][code];
        } else {
            const isUnknownErrorsExists = Object.keys(errors).some((errorKey) => !errorMessages[errorKey]);

            if (fieldName === 'form' && isUnknownErrorsExists) {
                errorMessage = errorMessages.form.internal;
            }
        }

        if (!errorMessage) {
            return null;
        }

        return (
            <div className='control-error-block js-error'>
                {errorMessage}
            </div>
        );
    }

    render() {
        const {title, description, homepage, ios_app_id, ios_appstore_url, scopes, is_yandex, platforms,
            android_package_name, android_cert_fingerprints, android_appstore_url, redirect_uri, turboapp_base_url} = this.props.form;
        const {
            csrf,
            dispatch,
            clientId,
            openedScope,
            allScopes,
            shouldBeModerated,
            tokenLifeSpan,
            loading,
            invalidateTokens,
            customUrlSchemes,
            universalLinkDomains,
            showWarning,
            isCorporate,
            icon,
            retpath,
            isTokenRefreshable
        } = this.props;
        const iosSelected = platforms.includes('ios');
        const webSelected = platforms.includes('web');
        const androidSelected = platforms.includes('android');
        const turboappSelected = platforms.includes('turboapp');
        const supportLink = OAUTH_SUPPORT_LINK.replace('%tld%', this.domain);
        const supportLinkHead = `<a class='link link_js_inited link_theme_islands'
                                    href=${supportLink} target='_blank'>`;
        const supportLinkTail = '</a>';

        return (
            <div className='edit-page'>
                <form onSubmit={this.onFormSubmit} ref='form'>
                    <div className='edit-page__menu'>
                        <a href='/' className='edit-page__menu-item'>
                            {i18n('frontend.app.app.registered-apps')}
                        </a>
                        <div className='edit-page__menu-item edit-page__menu-item_selected'>
                            {clientId ? i18n('frontend.app.app.edit.title') : i18n('frontend.app.app.create.title')}
                        </div>
                    </div>
                    <header className='edit-page__header'>
                        <h2>
                            {clientId ? i18n('frontend.app.app.edit.title') : i18n('frontend.app.app.create.title')}
                        </h2>
                        <div className='edit-page__subheader' dangerouslySetInnerHTML={{
                            __html: i18n('frontend.app.app.support-link', supportLinkHead, supportLinkTail)
                        }}/>
                    </header>
                    <div className='edit-page__field-block'>
                        <label className='edit-page__field-label'>
                            {i18n('frontend.access.access.tokens.apppass.list-title')}*:
                        </label>
                        <div className='edit-page__field-input'>
                            <TextInput
                                theme='normal'
                                text={title}
                                type='text'
                                size='m'
                                onChange={this.onTextFieldChange}
                                onBlur={this.onFieldBlur}
                                onFocus={this.onFieldFocus}
                                name='title'/>
                        </div>
                        {this.renderError('title')}
                        <div className='edit-page__field-description'>
                            {i18n('frontend.app.app.title.hint')}
                        </div>
                    </div>
                    <If condition={isCorporate}>
                        <div className='edit-page__field-block'>
                            <Checkbox
                                theme='normal'
                                className='edit-page__checkbox'
                                onChange={this.onYandexAppCheckboxChange}
                                checked={is_yandex}
                                name='is_yandex'
                                id="corporate-is-yandex"
                                size='m'>
                                <span className='edit-page__section-title'>{i18n('frontend.fields.field.yandexAppFlag.label')}</span>
                            </Checkbox>
                        </div>
                    </If>
                    <div className='edit-page__field-block'>
                        <label className='edit-page__field-label'>
                            {i18n('frontend.app.app.description')}:
                        </label>
                        <TextArea 
                            theme='islands'
                            size='l'
                            text={description}
                            name='description'
                            onFocusChange={this.onTextAreaFocus}
                            cls='edit-page__field-input edit-page__field-textarea'
                            onChange={this.onTextFieldChange}
                            rows={2}
                        />
                        {this.renderError('description')}
                        <div className='edit-page__field-description'>
                            {i18n('frontend.app.app.description.hint')}
                        </div>
                    </div>
                    <div className='edit-page__field-block edit-page__icon-block'>
                        <label className='edit-page__field-label'>
                            {i18n('frontend.app.app.icon.title')}:
                        </label>
                        <Choose>
                            <When condition={icon && icon.includes('http')}>
                                <div className='edit-page__icon'>
                                    <img src={icon} width='50' height='50'/>
                                </div>
                            </When>
                            <When condition={icon}>
                                <div className='edit-page__icon'>
                                    <div className='edit-page__icon-text'>{icon}</div>
                                </div>
                            </When>
                            <Otherwise>
                                <div className='edit-page__icon'>
                                    {i18n('frontend.fields.field.icon.icon-not-selected')}
                                </div>
                            </Otherwise>
                        </Choose>
                        <If condition={icon}>
                            <Button theme='islands'
                                size='l'
                                type='button'
                                cls='edit-page__field-button'
                                onClick={this.onDeleteIconClick}
                                view='pseudo'>
                                {i18n('frontend.fields.field.icon.remove')}
                            </Button>
                        </If>
                        <FileInput
                            ref='icon'
                            tabIndex={3}
                            name='icon_file'
                            className='edit-page__field-button'
                            title={i18n('frontend.common.common.upload')}
                            onChange={this.onFileUpload}/>
                        {this.renderError('icon')}
                        {this.renderError('icon_file')}
                        <div className='edit-page__field-description'>
                            {i18n('frontend.app.app.icon.hint')}
                        </div>
                    </div>
                    <div className='edit-page__field-block'>
                        <label className='edit-page__field-label'>
                            {i18n('frontend.app.app.homepage.title')}
                        </label>
                        <div className='edit-page__field-input'>
                            <TextInput
                                theme='normal'
                                size='m'
                                name='homepage'
                                maxLength={1024}
                                onChange={this.onTextFieldChange}
                                onBlur={this.onFieldBlur}
                                onFocus={this.onFieldFocus}
                                text={homepage}/>
                        </div>
                        {this.renderError('homepage')}
                        <div className='edit-page__field-description'>
                            {i18n('frontend.app.app.homepage.hint')}
                        </div>
                    </div>
                    <header className='edit-page__header edit-page__platforms'>
                        <h3>{i18n('frontend.app.app.platforms.title')}</h3>
                        <div className='edit-page__subheader'>
                            {i18n('frontend.app.app.platforms.hint')}
                        </div>
                    </header>
                    <div className='edit-page__section'>
                        <div>
                            <Checkbox
                                theme='normal'
                                className='edit-page__checkbox'
                                size='m'
                                name='ios'
                                id="platform-ios"
                                checked={iosSelected}
                                onChange={this.onSectionCheck}>
                                <span className='edit-page__section-title'>{i18n('frontend.app.app.ios.title')}</span>
                            </Checkbox>
                        </div>
                        {iosSelected && (
                            <div>
                                {ios_app_id.map((appId, index) => (
                                    <div className='edit-page__field-block'>
                                        <label className='edit-page__field-label'>
                                            {i18n('frontend.app.app.ios.bundle.title')} #{index + 1}:
                                        </label>
                                        <div className='edit-page__field-input edit-page__ios-app-id-block'>
                                            <div className='edit-page__field-input'>
                                                <TextInput
                                                    theme='normal'
                                                    size='m'
                                                    name={`ios_app_id.${index}`}
                                                    text={appId}
                                                    onChange={this.onTextFieldChange}
                                                    onBlur={this.onFieldBlur}
                                                    onFocus={this.onFieldFocus}
                                                />
                                            </div>
                                            <DeleteButton name='ios_app_id' index={index} dispatch={dispatch}/>
                                            <Button theme='islands'
                                                size='l'
                                                view='pseudo'
                                                cls={classnames('edit-page__field-button', {
                                                    'edit-page__field-button_hidden': 
                                                        ios_app_id.length === 5 || index !== ios_app_id.length - 1
                                                })}
                                                disabled={ios_app_id.some(id => !id.length)}
                                                name='ios_app_id'
                                                onClick={this.onAddButtonClick}>
                                                {i18n('frontend.common.common.add')}
                                            </Button>
                                        </div>
                                        {this.renderError('ios_app_id', index)}
                                        {!index && (
                                            <div className='edit-page__field-description'>
                                                {i18n('frontend.app.app.ios.bundle.hint')}
                                            </div>
                                        )}
                                    </div>
                                ))}
                                <div className='edit-page__field-block'>
                                    <label className='edit-page__field-label'>
                                        {i18n('frontend.app.app.ios.appstore.title')}:
                                    </label>
                                    <div className='edit-page__field-input'>
                                        <TextInput
                                            theme='normal'
                                            size='m'
                                            name='ios_appstore_url'
                                            onChange={this.onTextFieldChange}
                                            onBlur={this.onFieldBlur}
                                            onFocus={this.onFieldFocus}
                                            text={ios_appstore_url}/>
                                    </div>
                                    {this.renderError('ios_appstore_url')}
                                    <div className='edit-page__field-description'>
                                        {i18n('frontend.app.app.ios.appstore.hint')}
                                    </div>
                                </div>
                                {universalLinkDomains && (
                                    <div className='edit-page__field-block'>
                                        <label className='edit-page__field-label'>
                                            {i18n('frontend.app.app.ios.links.title')}:
                                        </label>
                                        {universalLinkDomains.map((universalLinkDomain) => (
                                            <div className='edit-page__field-input' key={universalLinkDomain}>
                                                <div className='edit-page__field-input_disabled'>
                                                    {universalLinkDomain}
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                )}
                            </div>
                        )}
                    </div>
                    <div className='edit-page__section'>
                        <div>
                            <Checkbox
                                theme='normal'
                                size='m'
                                name='android'
                                id="platform-android"
                                className='edit-page__checkbox'
                                checked={androidSelected}
                                onChange={this.onSectionCheck}>
                                <span className='edit-page__section-title'>
                                    {i18n('frontend.app.app.android.title')}
                                </span>
                            </Checkbox>
                        </div>
                        {androidSelected && (
                            <div>
                                {android_package_name.map((androidPackageName, index) => (
                                    <div className='edit-page__field-block edit-page__android-package-name-block'>
                                        <label className='edit-page__field-label'>
                                            {i18n('frontend.app.app.android.package.title')} #{index + 1}:
                                        </label>
                                        <div className='edit-page__field-input'>
                                            <TextInput
                                                theme='normal'
                                                size='m'
                                                name={`android_package_name.${index}`}
                                                onChange={this.onTextFieldChange}
                                                onBlur={this.onFieldBlur}
                                                onFocus={this.onFieldFocus}
                                                text={androidPackageName}/>
                                        </div>
                                        <DeleteButton name='android_package_name' index={index} dispatch={dispatch}/>
                                        <Button theme='islands'
                                            size='l'
                                            view='pseudo'
                                            cls={classnames('edit-page__field-button', {
                                                'edit-page__field-button_hidden': 
                                                    android_package_name.length === 5 || index !== android_package_name.length - 1
                                            })}
                                            disabled={android_package_name.some(id => !id.length)}
                                            name='android_package_name'
                                            onClick={this.onAddButtonClick}>
                                            {i18n('frontend.common.common.add')}
                                        </Button>
                                        {this.renderError('android_package_name', index)}
                                        {!index && (
                                            <div className='edit-page__field-description'>
                                                {i18n('frontend.app.app.android.package.hint')}
                                            </div>
                                        )}
                                    </div>
                                ))}
                                <div className='edit-page__field-block'>
                                    <label className='edit-page__field-label'>
                                        {i18n('frontend.app.app.android.appstore.title')}:
                                    </label>
                                    <div className='edit-page__field-input'>
                                        <TextInput
                                            theme='normal'
                                            size='m'
                                            name='android_appstore_url'
                                            onChange={this.onTextFieldChange}
                                            onBlur={this.onFieldBlur}
                                            onFocus={this.onFieldFocus}
                                            text={android_appstore_url}/>
                                    </div>
                                    {this.renderError('android_appstore_url')}
                                    <div className='edit-page__field-description'>
                                        {i18n('frontend.app.app.android.appstore.hint')}
                                    </div>
                                </div>
                                {android_cert_fingerprints.map((fingerprint, index) => (
                                    <div>
                                        <div className='edit-page__field-block edit-page__fingerprint-block'>
                                            <label className='edit-page__field-label'>
                                                {`${i18n('frontend.app.app.android.fingerprints')} #${index + 1}:`}
                                            </label>
                                            <div className='edit-page__field-input'>
                                                <TextInput
                                                    theme='normal'
                                                    size='m'
                                                    name={`android_cert_fingerprints.${index}`}
                                                    placeholder='00:00:00:00:00:00:00:00:00:00:00:00:00:00:00:00:00:00:00:00:00:00:00:00:00:00:00:00:00:00:00:00'
                                                    text={fingerprint}
                                                    onChange={this.onTextFieldChange}
                                                    onBlur={this.onFieldBlur}
                                                    onFocus={this.onFieldFocus}/>
                                            </div>
                                            <DeleteButton name='android_cert_fingerprints' index={index} dispatch={dispatch}/>
                                            <Button theme='islands'
                                                size='l'
                                                view='pseudo'
                                                cls={classnames('edit-page__field-button', {
                                                    'edit-page__field-button_hidden':
                                                    index !== android_cert_fingerprints.length - 1 ||
                                                    android_cert_fingerprints.length === 5
                                                })}
                                                name='android_cert_fingerprints'
                                                onClick={this.onAddButtonClick}>
                                                {i18n('frontend.common.common.add')}
                                            </Button>
                                        </div>
                                        {this.renderError('android_cert_fingerprints', index)}
                                    </div>
                                ))}
                                {customUrlSchemes && (
                                    <div className='edit-page__field-block'>
                                        <label className='edit-page__field-label'>
                                            {i18n('frontend.app.app.android.schemes.title')}:
                                        </label>
                                        {customUrlSchemes.map((urlScheme) => (
                                            <div className='edit-page__field-input' key={urlScheme}>
                                                <div className='edit-page__field-input_disabled'>
                                                    {urlScheme}
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                )}
                            </div>
                        )}
                    </div>
                    <div className='edit-page__section'>
                        <div>
                            <Checkbox
                                theme='normal'
                                className='edit-page__checkbox'
                                size='m'
                                name='web'
                                id="platform-web"
                                checked={webSelected}
                                onChange={this.onSectionCheck}>
                                <span className='edit-page__section-title'>
                                    {i18n('frontend.app.app.web.title')}
                                </span>
                            </Checkbox>
                        </div>
                        <If condition={webSelected}>
                            <For each='uri' index='index' of={redirect_uri}>
                                <div className='edit-page__field-block edit-page__redirect-block'>
                                    <label className='edit-page__field-label'>
                                        {`${i18n('frontend.app.app.web.callbackuri')} #${index + 1}:`}
                                    </label>
                                    <div className='edit-page__field-input'>
                                        <TextInput
                                            theme='normal'
                                            size='m'
                                            name={`redirect_uri.${index}`}
                                            text={uri}
                                            maxLength={1024}
                                            onChange={this.onTextFieldChange}
                                            onBlur={this.onFieldBlur}
                                            onFocus={this.onFieldFocus}/>
                                    </div>
                                    <DeleteButton name='redirect_uri' index={index} dispatch={dispatch} />
                                    <Button theme='islands'
                                        size='l'
                                        view='pseudo'
                                        cls={classnames('edit-page__field-button', {
                                            'edit-page__field-button_hidden':
                                            index !== redirect_uri.length - 1 ||
                                            redirect_uri.length === 20
                                        })}
                                        disabled={redirect_uri.some(uri => !uri.length)}
                                        name='redirect_uri'
                                        onClick={this.onAddButtonClick}>
                                        {i18n('frontend.common.common.add')}
                                    </Button>
                                </div>
                                <If condition={index === 0}>
                                    <div className='edit-page__field-description'>
                                        {i18n('frontend.app.app.debug.hint')}
                                    </div>
                                </If>
                                {this.renderError('redirect_uri', index)}
                                <div className='edit-page__field-description'>
                                    <Link theme='islands' size='m' onClick={this.setURLForDevelopment.bind(null, index)}>
                                        {i18n('frontend.app.app.debug.title')}
                                    </Link>
                                </div>
                            </For>
                        </If>
                    </div>
                    <div className='edit-page__section'>
                        <div>
                            <Checkbox
                                theme='normal'
                                className='edit-page__checkbox'
                                size='m'
                                name='turboapp'
                                id="platform-turboapp"
                                checked={turboappSelected}
                                onChange={this.onSectionCheck}>
                                <span className='edit-page__section-title'>
                                    {i18n('frontend.app.app.turboapp.title')}
                                </span>
                            </Checkbox>
                            <If condition={turboappSelected}>
                                <div className='edit-page__field-block'>
                                    <label className='edit-page__field-label'>
                                        {i18n('frontend.app.app.turboapp.baseurl.title')}:
                                    </label>
                                    <div className='edit-page__field-input'>
                                        <TextInput
                                            theme='normal'
                                            size='m'
                                            name='turboapp_base_url'
                                            onChange={this.onTextFieldChange}
                                            onBlur={this.onFieldBlur}
                                            onFocus={this.onFieldFocus}
                                            text={turboapp_base_url}/>
                                    </div>
                                    {this.renderError('turboapp_base_url')}
                                    <div className='edit-page__field-description'>
                                        {i18n('frontend.app.app.turboapp.baseurl.hint')}
                                    </div>
                                </div>
                            </If>
                        </div>
                    </div>
                    <header className='edit-page__header edit-page__scopes-header'>
                        <h3>{i18n('frontend.app.app.permissions.title')}*</h3>
                        <div className='edit-page__subheader'>
                            {i18n('frontend.app.app.permissions.hint')}
                        </div>
                        {this.renderError('scopes')}
                    </header>
                    <div>
                        <For each='scope' of={allScopes}>
                            <Scope
                                key={scope.id}
                                scope={scope}
                                isOpen={openedScope === scope.id}
                                allSelectedPermissions={scopes}
                                dispatch={dispatch}/>
                        </For>
                    </div>
                    <header className='edit-page__header edit-page__access-header'>
                        <h3>{i18n('frontend.app.app.params')}</h3>
                        <div className='edit-page__app-params'>
                            <div className='edit-page__app-params-title'>{i18n('frontend.app.app.moderation')}:</div>
                            <div>{shouldBeModerated ? i18n('frontend.app.app.approval.required') :
                                i18n('frontend.app.app.approval.not_required')}
                            </div>
                        </div>
                        <If condition={tokenLifeSpan !== undefined}>
                            <div className='edit-page__app-params'>
                                <div className='edit-page__app-params-title'>
                                    {i18n('frontend.app.app.tokenlifespan')}:
                                </div>
                                <div>
                                    {isTokenRefreshable && tokenLifeSpan &&
                                        `${i18n('frontend.app.app.ttl.at_least')} `
                                    }
                                    {EditPage.formatTokenLifeSpan(tokenLifeSpan)}
                                </div>
                            </div>
                        </If>
                    </header>
                    <div className='edit-page__controls'>
                        <Choose>
                            <When condition={shouldBeModerated}>
                                <div className='icon-warning'/>
                                <div className='warning warning-text'>
                                    {i18n('frontend.app.app.should_be_moderated.warning')}
                                </div>
                            </When>
                            <When condition={invalidateTokens && clientId}>
                                <div className='icon-warning'/>
                                <div className='warning warning-text'>
                                    {i18n('frontend.app.app.revoke_tokens.warning')}
                                </div>
                            </When>
                            <When condition={!invalidateTokens && clientId}>
                                <div className='icon-success'/>
                                <div className='success success-text'>
                                    {i18n('frontend.app.app.revoke_tokens.success')}
                                </div>
                            </When>
                            <Otherwise>
                                <div/>
                            </Otherwise>
                        </Choose>
                        <div className='edit-page__save-button'>
                            <Choose>
                                <When condition={!loading}>
                                    <Button theme='islands' size='l' view='action' type='submit'>
                                        {clientId ? i18n('frontend.app.app.save_changes') :
                                            i18n('frontend.app.app.create_application')}
                                    </Button>
                                </When>
                                <Otherwise>
                                    <Spinner size='m' progress='yes'/>
                                </Otherwise>
                            </Choose>
                        </div>
                    </div>
                </form>
                <div className='edit-page__form-error'>
                    {this.renderError('form')}
                </div>
                <If condition={clientId}>
                    <form className='edit-page__generate-password'
                        action={`/client/password/reset/${clientId}`} method='post'>
                        <input type='hidden' name='csrf' value={csrf}/>
                        <input type='hidden' name='retpath' value={retpath}/>
                        <Button theme='islands' size='l' view='pseudo' type='submit'>
                            {i18n('frontend.app.app.edit.new_secret')}
                        </Button>
                    </form>
                </If>
                <Modal theme='islands' size='m' visible={showWarning}>
                    <div className='edit-page__modal'>
                        <h2 className='edit-page__modal-title'>
                            {i18n('frontend.app.app.update_application')}
                        </h2>
                        <div className='edit-page__modal-description'>
                            <div className='icon-warning'/>
                            <div>
                                {i18n('frontend.app.app.revoke_tokens.warning')}!
                            </div>
                        </div>
                        <div className='edit-page__modal-controls'>
                            <Button theme='islands' size='l' view='pseudo' onClick={this.hideWarning}>
                                {i18n('frontend.common.common.undo')}
                            </Button>
                            <Choose>
                                <When condition={!loading}>
                                    <Button theme='islands' size='l' view='action' onClick={this.onSaveButtonClick}>
                                        {i18n('frontend.app.app.save_changes')}
                                    </Button>
                                </When>
                                <Otherwise>
                                    <Spinner size='m'/>
                                </Otherwise>
                            </Choose>
                        </div>
                    </div>
                </Modal>
            </div>
        );
    }
}

EditPage.propTypes = {
    dispatch: PropTypes.func.isRequired,
    errors: PropTypes.object.isRequired,
    tld: PropTypes.string.isRequired,
    lang: PropTypes.string.isRequired,
    clientId: PropTypes.string,
    csrf: PropTypes.string.isRequired,
    retpath: PropTypes.string.isRequired,
    form: PropTypes.shape({
        title: PropTypes.string,
        description: PropTypes.string,
        homepage: PropTypes.string,
        is_yandex: PropTypes.bool,
        icon_id: PropTypes.string,
        icon_file: PropTypes.object,
        platforms: PropTypes.arrayOf(PropTypes.string).isRequired,
        ios_app_id: PropTypes.arrayOf(PropTypes.string),
        ios_appstore_url: PropTypes.string,
        android_package_name: PropTypes.arrayOf(PropTypes.string),
        android_appstore_url: PropTypes.string,
        android_cert_fingerprints: PropTypes.arrayOf(PropTypes.string),
        redirect_uri: PropTypes.arrayOf(PropTypes.string),
        scopes: PropTypes.array.isRequired
    }).isRequired,
    isCorporate: PropTypes.bool.isRequired,
    customUrlSchemes: PropTypes.arrayOf(PropTypes.string),
    universalLinkDomains: PropTypes.arrayOf(PropTypes.string),
    icon: PropTypes.string,
    iconURL: PropTypes.string,
    openedScope: PropTypes.string,
    allScopes: PropTypes.array.isRequired,
    shouldBeModerated: PropTypes.bool.isRequired,
    tokenLifeSpan: PropTypes.number,
    invalidateTokens: PropTypes.bool.isRequired,
    showWarning: PropTypes.bool.isRequired
};
// title: application.title || '',
//     description: application.description || '',
//     icon_id: application.icon_id || '',
//     icon_file: undefined,
//     is_yandex: state.isCorporate ? application.is_yandex : undefined,
//     homepage: application.homepage || '',
//     ios_app_id: ios.app_id,
//     ios_appstore_url: ios.appstore_url,
//     android_package_name: android.package_name,
//     android_cert_fingerprints: android.cert_fingerprints,
//     android_appstore_url: android.appstore_url,
//     redirect_uri: web.redirect_uris,
//     scopes: flattenScopes(application.scopes),
//     platforms: Object.keys(application.platforms)

class Scope extends Component {
    constructor(props) {
        super(props);

        this.toggleScopePermissions = this.toggleScopePermissions.bind(this);
    }

    shouldComponentUpdate(newProps) {
        const {allSelectedPermissions, isOpen} = this.props;
        const newIsOpen = newProps.isOpen;

        // Permissions changed
        return newIsOpen && allSelectedPermissions !== newProps.allSelectedPermissions || newIsOpen !== isOpen;
    }

    toggleScopePermissions() {
        const {dispatch, scope} = this.props;

        dispatch(toggleScopePermissions(scope.id));
    }

    render() {
        const {dispatch, scope, allSelectedPermissions, isOpen} = this.props;
        const selectedPermissions = allSelectedPermissions.filter(function(permission) {
            return scope.permissions.some((scopePermission) => {
                return scopePermission.id === permission.name;
            });
        });

        return (
            <div className={`edit-page__scope ${isOpen ? 'edit-page__scope_opened' : ''}`}>
                <header className='edit-page__scope-header'
                    onClick={this.toggleScopePermissions}>
                    <h4 className='edit-page__scope-title'>
                        <span>{scope.section} </span>
                        <If condition={!isOpen && selectedPermissions.length > 0}>
                            {`(${i18n('frontend.app.app.selected')}: ${selectedPermissions.length})`}
                        </If>
                    </h4>
                    <div className='icon-arrow edit-page__icon-arrow'/>
                </header>
                <If condition={isOpen}>
                    <div>
                        <For each='permission' of={scope.permissions}>
                            <Permission
                                key={permission.id}
                                dispatch={dispatch}
                                permission={permission}
                                selected={Boolean(selectedPermissions.find((p) => p.name === permission.id))}/>
                        </For>
                    </div>
                </If>
            </div>
        );
    }
}

Scope.propTypes = {
    dispatch: PropTypes.func.isRequired,
    scope: PropTypes.shape({
        id: PropTypes.string.isRequired,
        section: PropTypes.string.isRequired,
        permissions: PropTypes.array.isRequired
    }).isRequired,
    isOpen: PropTypes.bool.isRequired,
    allSelectedPermissions: PropTypes.arrayOf(PropTypes.object)
};

class Permission extends React.PureComponent {
    constructor(props) {
        super(props);

        this.onPermissionClick = this.onPermissionClick.bind(this);
        this.onLabelClick = this.onLabelClick.bind(this);
    }

    onPermissionClick(event) {
        const {dispatch, permission} = this.props;
        const {id, section} = permission;

        dispatch(selectPermission(event.target.checked, section, id));
        dispatch(validateApplicationChanges('scopes'));
    }

    onLabelClick() {
        this.onPermissionClick(!this.props.selected);
    }

    render() {
        const {permission, selected} = this.props;

        return (
            <div className='edit-page__permission'>
                <Checkbox
                    theme='normal'
                    className='edit-page__checkbox'
                    onChange={this.onPermissionClick}
                    checked={selected}
                    section={permission.section}
                    value={permission.id}
                    name={`scopes:${permission.id}`}
                    id={`scopes-${(permission.id || '').replace(':', '-')}`}
                    size='m'>
                    <span className='edit-page__section-title'>{permission.title}</span>
                </Checkbox>
            </div>
        );
    }
}

Permission.propTypes = {
    dispatch: PropTypes.func.isRequired,
    permission: PropTypes.shape({
        id: PropTypes.string.isRequired,
        section: PropTypes.string.isRequired,
        title: PropTypes.string.isRequired
    }).isRequired,
    selected: PropTypes.bool.isRequired
};

class FileInput extends Component {
    constructor(props) {
        super(props);

        this.onButtonPressed = this.onButtonPressed.bind(this);
    }

    clear() {
        this.refs.file.value = '';
    }

    onButtonPressed() {
        this.refs.file.click();
    }

    render() {
        const {name, title, className} = this.props;

        return (
            <div onKeyPress={this.onButtonPressed}
                className={`file-input ${className}`}>
                <Button theme='islands' type='button' view='pseudo' size='l' onClick={this.onButtonPressed}>
                    {title}
                </Button>
                <input type='file'
                    tabIndex='-1'
                    ref='file'
                    name={name}
                    className='file-input__input'
                    onChange={this.props.onChange}/>
            </div>
        );
    }
}

class DeleteButton extends Component {
    onClick = (event) => {
        const {name, index, dispatch} = this.props;

        event.preventDefault();
        dispatch(deleteItem(name, index));
    }

    render() {
        const {index} = this.props;

        return (
            <Button theme='islands'
                size='l'
                view='pseudo'
                cls={`edit-page__field-button${!index ? ' edit-page__field-button_hidden' : ''}`}
                onClick={this.onClick}>
                {i18n('frontend.common.common.delete')}
            </Button>
        )
    }
}

DeleteButton.propTypes = {
    name: PropTypes.string.isRequired,
    index: PropTypes.number.isRequired,
    dispatch: PropTypes.func.isRequired
}

FileInput.propTypes = {
    name: PropTypes.string.isRequired,
    title: PropTypes.string.isRequired,
    className: PropTypes.string,
    tabIndex: PropTypes.number
};

FileInput.defaultProps = {
    tabIndex: 0,
    className: '',
    onChange: function() {} // eslint-disable-line
};

export default connect(EditPage.mapStateToProps)(EditPage);
