import u from 'updeep';
import {
    APP_INITIALIZED
} from '../actions/app';

import {
    EDIT_PAGE_UPDATE_TEXT_FIELD,
    EDIT_PAGE_UPDATE_SECTION,
    EDIT_PAGE_SAVE_APPLICATION,
    EDIT_PAGE_SAVE_APPLICATION_SUCCESS,
    EDIT_PAGE_SAVE_APPLICATION_FAIL,
    EDIT_PAGE_ADD_ITEM,
    EDIT_PAGE_DELETE_ITEM,
    EDIT_PAGE_TOGGLE_SCOPE_PERMISSIONS,
    EDIT_PAGE_SELECT_PERMISSION,
    EDIT_PAGE_SELECT_ICON,
    EDIT_PAGE_DELETE_ICON,
    EDIT_PAGE_VALIDATE_APPLICATION_SUCCESS,
    EDIT_PAGE_VALIDATE_APPLICATION_FAIL,
    EDIT_PAGE_SHOW_WARNING,
    EDIT_PAGE_SET_FIELD_ERROR,
    EDIT_PAGE_REMOVE_ERROR
} from '../actions/edit';

const ONE_MEGABYTE = 1048576;

// Helper function

function addItem(items) {
    return [].concat(items, '');
}

function deleteItem(items = [], index) {
    return [].concat(items.slice(0, index), items.slice(index + 1, items.length));
}

function shouldBeModerated(selectedPermissions) {
    if (!selectedPermissions) {
        return false;
    }

    return selectedPermissions.some(function(p) {
        return p.requiresApproval;
    });
}

function maxTokenLifeSpan(selectedPermissions) {
    let maxTokenLifeSpan = Number.POSITIVE_INFINITY; // eslint-disable-line

    if (!selectedPermissions) {
        return undefined;
    }

    selectedPermissions.forEach(function(p) {
        let ttl = p.ttl;

        if (ttl === null || ttl === 'INFINITE') {
            ttl = Number.POSITIVE_INFINITY;
        }

        if (maxTokenLifeSpan > ttl && Number.isFinite(ttl)) {
            maxTokenLifeSpan = ttl;
        }
    });

    return maxTokenLifeSpan === Number.POSITIVE_INFINITY ? null : maxTokenLifeSpan;
}

function isTokenRefreshable(selectedPermissions) {
    if (!selectedPermissions) {
        return false;
    }

    return selectedPermissions.every(permission =>
        permission.ttlRefreshable || !permission.ttl || permission.ttl === 'INFINITE'
    );
}

function flattenScopes(scopes = {}) {
    const permissions = [];

    Object.keys(scopes).forEach((scopeName) => {
        const scope = scopes[scopeName];

        Object.keys(scope).forEach((permissionName) => {
            const permission = scope[permissionName];

            permissions.push({
                name: permissionName,
                section: scopeName,
                requiresApproval: permission.requires_approval,
                ttl: permission.ttl,
                ttlRefreshable: permission.is_ttl_refreshable
            });
        });
    });

    return permissions;
}

function formatErrors(errors) {
    const formattedErrors = {};

    errors.forEach(function(code) {
        const splitCode = code.split('.');
        let fieldName;
        let error;
        let index;

        if (splitCode.length === 3) {
            fieldName = splitCode[0];
            index = splitCode[1];
            error = splitCode[2];
            formattedErrors[fieldName] = formattedErrors[fieldName] || [];
            formattedErrors[fieldName][index] = error;
        } else {
            fieldName = splitCode[0];
            error = splitCode[1];
            formattedErrors[fieldName] = error;
        }
    });

    return formattedErrors;
}

export function editPage(state = {}, action) {
    let newState;

    switch (action.type) {
        case APP_INITIALIZED: {
            const application = state.application || {};
            const platforms = application.platforms || {};
            const android = platforms.android || {};
            const ios = platforms.ios || {};
            const web = platforms.web || {};
            const turboapp = platforms.turboapp || {};

            const form = {
                title: application.title || '',
                description: application.description || '',
                icon_id: application.icon_id || '',
                icon_file: undefined,
                is_yandex: application.is_yandex || false,
                homepage: application.homepage || '',
                ios_app_id: ios.app_ids,
                ios_appstore_url: ios.appstore_url,
                android_package_name: android.package_names,
                android_cert_fingerprints: android.cert_fingerprints,
                android_appstore_url: android.appstore_url,
                redirect_uri: web.redirect_uris,
                turboapp_base_url: turboapp.base_url,
                scopes: flattenScopes(application.scopes),
                platforms: Object.keys(platforms)
            };

            newState = {
                retpath: state.retpath,
                form,
                csrf: state.csrf,
                errors: {},
                showWarning: false,
                customUrlSchemes: android.custom_urlschemes,
                universalLinkDomains: ios.universal_link_domains,
                loading: false,
                clientId: state.clientId || '',
                invalidateTokens: false,
                shouldBeModerated: shouldBeModerated(form.scopes),
                tokenLifeSpan: maxTokenLifeSpan(form.scopes),
                isTokenRefreshable: isTokenRefreshable(form.scopes),
                icon: application.icon || application.icon_url || '',
                iconURL: application.icon || application.icon_url || '',
                allScopes: state.allScopes,
                isCorporate: state.isCorporate,
                openedScope: null
            };
            break;
        }
        case EDIT_PAGE_UPDATE_TEXT_FIELD: {
            const value = action.value;
            const [name, index] = action.name.split('.');
            const form = {};

            if (index === undefined) {
                form[name] = value;
            } else {
                form[name] = state.form[name].slice();
                form[name][index] = value;
            }

            newState = u({
                form
            }, state);
            break;
        }
        case EDIT_PAGE_UPDATE_SECTION: {
            let updates = {
                form: {}
            };
            const section = action.section;
            const platforms = state.form.platforms.slice();
            const show = !platforms.includes(section);

            if (show) {
                platforms.push(section);
            } else {
                platforms.splice(platforms.findIndex((p) => p === section), 1);
            }

            switch (section) {
                case 'ios': {
                    if (show) {
                        updates = {
                            form: {
                                ios_app_id: [''],
                                ios_appstore_url: ''
                            },
                            errors: u.omit(['ios_app_id', 'ios_appstore_url']),
                            universalLinkDomains: undefined
                        };
                    } else {
                        updates = {
                            form: u.omit(['ios_app_id', 'ios_appstore_url'])
                        };
                    }
                    break;
                }
                case 'android': {
                    const errors = [
                        'android_package_name',
                        'android_appstore_url',
                        'android_cert_fingerprints'
                    ];

                    if (show) {
                        updates = {
                            form: {
                                android_package_name: [''],
                                android_appstore_url: '',
                                android_cert_fingerprints: ['']
                            },
                            errors: u.omit(errors),
                            customUrlSchemes: undefined
                        };
                    } else {
                        updates = {
                            form: u.omit([
                                'android_package_name',
                                'android_appstore_url',
                                'android_cert_fingerprints'
                            ])
                        };
                    }
                    break;
                }
                case 'web': {
                    const errors = ['redirect_uri'];

                    if (show) {
                        updates = {
                            form: {
                                redirect_uri: ['']
                            },
                            errors: u.omit(errors)
                        };
                    } else {
                        updates = {
                            form: u.omit('redirect_uri')
                        };
                    }
                    break;
                }
                case 'turboapp': {
                    const field = 'turboapp_base_url'
                    if (show) {
                        updates = {
                            form: {
                                [field]: ''
                            },
                            errors: u.omit([field])
                        }
                    } else {
                        updates = {
                            form: u.omit([field])
                        };
                    }
                    break;
                }
                default:
                    throw Error('Unexpected section name');
            }

            updates = u(updates, state);
            newState = u({
                form: {
                    platforms: u.constant(platforms)
                }
            }, updates);
            break;
        }
        case EDIT_PAGE_DELETE_ITEM: {
            const {name, index} = action;
            const form = {
                [name]: deleteItem(state.form[name], index)
            };
            const errors = {
                [name]: deleteItem(state.errors[name], index)
            };

            newState = u({
                form,
                errors
            }, state);
            break;
        }
        case EDIT_PAGE_ADD_ITEM: {
            const name = action.name;
            const form = {};
            let limit;

            switch (name) {
                case 'android_cert_fingerprints':
                case 'ios_app_id':
                case 'android_package_name': {
                    limit = 5;
                    break;
                }
                case 'redirect_uri': {
                    limit = 20;
                    break;
                }
                default:
                    throw Error('Attempt to add item to unknown section');
            }

            if (state.form[name].length === limit) {
                newState = state;
                break;
            }

            form[name] = addItem(state.form[name]);

            newState = u({
                form
            }, state);
            break;
        }
        case EDIT_PAGE_TOGGLE_SCOPE_PERMISSIONS: {
            const sectionId = action.sectionId;

            newState = u({
                openedScope: state.openedScope === sectionId ? null : sectionId
            }, state);
            break;
        }
        case EDIT_PAGE_SELECT_PERMISSION: {
            const {tokenLifeSpan, clientId, allScopes, form} = state;
            const prevScopes = form.scopes;
            const checked = action.checked;
            const permissionName = action.permission;
            const sectionName = action.section;
            const permission = allScopes
                .find((s) => sectionName === s.section).permissions
                .find((p) => permissionName === p.id);
            const newPermission = {
                name: permissionName,
                section: sectionName,
                requiresApproval: permission.requiresApproval,
                ttl: permission.ttl,
                ttlRefreshable: permission.ttlRefreshable
            };
            const scopes = checked ? prevScopes.concat(newPermission) :
                prevScopes.filter((p) => p.name !== permissionName);

            newState = u({
                shouldBeModerated: shouldBeModerated(scopes),
                tokenLifeSpan: clientId ? tokenLifeSpan : maxTokenLifeSpan(scopes),
                isTokenRefreshable: isTokenRefreshable(scopes),
                form: {
                    scopes
                }
            }, state);
            break;
        }
        case EDIT_PAGE_SELECT_ICON: {
            const file = action.file;
            const icon = state.iconURL || '';
            const notImage = !file.type.match(/png|jpg|jpeg|gif/i);

            if (file.size > ONE_MEGABYTE || notImage) {
                newState = u({
                    icon,
                    form: {
                        icon_file: undefined
                    },
                    errors: {
                        icon: undefined,
                        icon_file: notImage ? 'invalid_format' : 'too_large'
                    }
                }, state);
            } else {
                newState = u({
                    icon: file.name,
                    errors: u.omit(['icon', 'icon_file']),
                    form: {
                        icon_file: file
                    }
                }, state);
            }
            break;
        }
        case EDIT_PAGE_DELETE_ICON: {
            const form = state.form;

            if (form.icon_file) {
                newState = u({
                    icon: state.iconURL,
                    form: u.omit('icon_file'),
                    errors: u.omit(['icon', 'icon_file'])
                }, state);
            } else if (form.icon_id) {
                newState = u({
                    icon: undefined,
                    iconURL: undefined,
                    form: u.omit(['icon_id', 'icon_file']),
                    errors: u.omit(['icon', 'icon_file'])
                }, state);
            } else {
                newState = state;
            }
            break;
        }
        case EDIT_PAGE_SAVE_APPLICATION: {
            newState = u({
                loading: true,
                showWarning: false
            }, state);
            break;
        }
        case EDIT_PAGE_SAVE_APPLICATION_SUCCESS: {
            newState = u({
                loading: false
            }, state);
            break;
        }
        case EDIT_PAGE_SAVE_APPLICATION_FAIL: {
            const errors = formatErrors(action.errors || []);
            const iconURL = state.iconURL || '';
            let icon = state.icon;
            let iconFile = state.form.icon_file;

            if (errors.icon === 'bad_format') {
                icon = iconURL;
                iconFile = undefined;
            }

            newState = u({
                errors: u.constant(errors),
                form: {
                    icon_file: iconFile
                },
                loading: false,
                icon
            }, state);
            break;
        }
        case EDIT_PAGE_VALIDATE_APPLICATION_SUCCESS: {
            const {invalidateTokens, tokenLifeSpan, showWarningIfNeeded, shouldBeModerated} = action;

            newState = u({
                invalidateTokens,
                tokenLifeSpan,
                shouldBeModerated,
                errors: u.constant({}),
                showWarning: showWarningIfNeeded && invalidateTokens || false
            }, state);
            break;
        }
        case EDIT_PAGE_VALIDATE_APPLICATION_FAIL: {
            newState = u({
                loading: false,
                errors: u.constant(formatErrors(action.errors))
            }, state);
            break;
        }
        case EDIT_PAGE_SHOW_WARNING: {
            newState = u({
                showWarning: action.show
            }, state);
            break;
        }
        case EDIT_PAGE_SET_FIELD_ERROR: {
            let {fieldName, error, index} = action;
            const parts = fieldName.split('.');
            let errors;

            if (parts.length === 2) {
                fieldName = parts[0];
                error = parts[1];
            } else if (parts.length === 3) {
                fieldName = parts[0];
                index = parts[1];
                error = parts[2];
            }

            if (index) {
                const oldErrors = state.errors[fieldName];

                errors = oldErrors && oldErrors.slice() || [];
                errors[index] = error;
            } else {
                errors = error;
            }

            newState = u({
                errors: {
                    [fieldName]: u.constant(errors)
                }
            }, state);
            break;
        }
        case EDIT_PAGE_REMOVE_ERROR: {
            const {errorKey} = action;

            newState = u({errors: u.omit([errorKey])}, state);
            break;
        }
        default:
            newState = state;
            break;
    }

    return newState;
}
