import React from 'react';
import {cn} from '@bem-react/classname';
import {Button} from 'pcomponents/Button';
import {Modal} from 'pcomponents/Modal';
import {Drawer} from 'pcomponents/Drawer';
import {ClientSideInfoLayout} from '../../common/ClientSideInfoLayout';
import {ClientNextStepsInfo} from '../../common/ClientNextStepsInfo';
import {ClientModerationMark} from '../../common/ClientModerationMark';
import {ROUTES} from '../../../const';
import {metric} from '../../../metric';
import {dateToText, getTtlInfoString} from '../../../../../utils';

import './ClientInfoPage.styl';

const b = cn('ClientInfoPage');

const SIDE_INFO_TEXTS = {
    clientId: {
        title: i18n('ClientInfo.side.clientId.title'),
        description: i18n('ClientInfo.side.clientId.description')
    },
    secret: {
        title: i18n('ClientInfo.side.secret.title'),
        description: i18n('ClientInfo.side.secret.description')
    },
    urls: {
        title: i18n('ClientInfo.side.urls.title'),
        description: i18n('ClientInfo.side.urls.description')
    }
};
/* const URLS_PREFIXES = {
    web: i18n('ClientInfo.side.prefix.web'),
    android: i18n('ClientInfo.side.prefix.android'),
    ios: i18n('ClientInfo.side.prefix.ios'),
    turboapp: i18n('ClientInfo.side.prefix.turboapp')
}; */
const BUTTONS_TEXTS = {
    load: i18n('ClientInfo.buttons.load'),
    edit: i18n('ClientInfo.buttons.edit'),
    delete: i18n('ClientInfo.buttons.delete'),
    password: i18n('ClientInfo.buttons.password')
};
const MODAL_DELETE_TEXTS = {
    title: i18n('ClientInfo.modal.title'),
    description: i18n('ClientInfo.modal.description'),
    web: i18n('ClientInfo.modal.platform.web'),
    ios: i18n('ClientInfo.modal.platform.ios'),
    android: i18n('ClientInfo.modal.platform.android'),
    turboapp: i18n('ClientInfo.modal.platform.turboapp')
};
const MODAL_PASSWORD_TEXTS = {
    title: i18n('ClientInfo.modal.password.title'),
    description: i18n('ClientInfo.modal.password.description')
};
const MODAL_BUTTONS_TEXTS = {
    reset: i18n('ClientInfo.modal.buttons.reset'),
    leave: i18n('ClientInfo.modal.buttons.leave'),
    cancel: i18n('ClientInfo.modal.buttons.cancel'),
    delete: i18n('ClientInfo.modal.buttons.delete')
};

const REQUESTS_TEXT = `!!!! Файл содержит ключи, будьте внимательны и передавайте файл только по защищенным каналам !!!

Запрос на oauth code (https://yandex.%tld%/dev/id/doc/dg/oauth/reference/web-client.html)
[…] – необязательные поля
https://oauth.yandex.%tld%/authorize?
response_type=code
& client_id=%client_id%
& redirect_uri=%redirect_uri%
[& device_id=<идентификатор устройства>]
[& device_name=<имя устройства>]
[& login_hint=<имя пользователя или электронный адрес>]
[& scope=<запрашиваемые необходимые права>]
[& optional_scope=<запрашиваемые опциональные права>]
[& force_confirm=yes]
[& state=<произвольная строка>]
[& display=popup]


Запрос на token(https://yandex.%tld%/dev/id/doc/dg/oauth/reference/auto-code-client.html#auto-code-client__get-token)
POST /token HTTP/1.1
Host: oauth.yandex.%tld%
Content-type: application/x-www-form-urlencoded
Content-Length: <длина тела запроса>
[Authorization: Basic <закодированная строка client_id:client_secret>]

grant_type=authorization_code
& code=<код подтверждения>
& client_id=%client_id%
& client_secret=%client_secret%
[& device_id=<идентификатор устройства>]
[& device_name=<имя устройства>]

(Необходимо указать или заголовок Authorization, или параметры client_id и client_secret)


Обновить token (https://yandex.%tld%/dev/id/doc/dg/oauth/reference/refresh-client.html#refresh-client__get-token)
POST /token HTTP/1.1
Host: oauth.yandex.%tld%
Content-type: application/x-www-form-urlencoded
Content-Length: <длина тела запроса>
[Authorization: Basic <закодированная строка client_id:client_secret>]

grant_type=refresh_token
 & refresh_token=<refresh_token>
[& client_id=%client_id%]
[& client_secret=%client_secret%]


Запрос данных о пользователе (https://yandex.%tld%/dev/id/doc/dg/api-id/reference/request.html)
GET https://login.yandex.%tld%/info?
  format=json | xml | jwt


Authorization: OAuth <OAuth-токен>`;

const downloadFile = (filename, text) => {
    const element = document.createElement('a');

    element.setAttribute('href', 'data:text/plain;charset=utf-8,' + encodeURIComponent(text));
    element.setAttribute('download', filename);

    element.style.display = 'none';
    document.body.appendChild(element);

    element.click();

    document.body.removeChild(element);
};

export class ClientInfoPage extends React.Component {
    state = {
        showDeletePopup: false,
        showPasswordPopup: false
    };
    componentDidMount() {
        const {clientId, clientIdUrl, editSetStateById, fetchClientsState} = this.props;

        if (clientId !== clientIdUrl) {
            editSetStateById(clientIdUrl);
        }

        fetchClientsState();
    }
    onLoadClick = () => {
        const {tld, client: {clientId, secret, callback} = {}} = this.props;

        metric.goal('lk_my_app_requests_click');

        downloadFile(
            'requests.txt',
            REQUESTS_TEXT.replace(/%tld%/g, tld)
                .replace(/%client_id%/g, clientId || '')
                .replace(/%client_secret%/g, secret || '')
                .replace(/%redirect_uri%/g, encodeURIComponent(callback || ''))
        );
    };

    onEditClick = () => {
        const {pushHistory, clientId} = this.props;

        metric.goal('lk_my_app_edit_click');
        pushHistory(ROUTES.FORM_EDIT.replace(':clientId', clientId));
    };
    onPasswordClick = () => {
        this.setState({showPasswordPopup: true});
    };
    onPasswordPopupClose = () => this.setState({showPasswordPopup: false});
    onConfirmPasswordResetClick = async () => {
        const {clientId, postClientPasswordReset, fetchClientsState, clientsUpdateClient} = this.props;
        const {status, secret, redirectTo} = await postClientPasswordReset(clientId);

        if (status === 'ok') {
            if (redirectTo) {
                window.location.href = redirectTo.replace('%retpath', encodeURIComponent(window.location.href));
                return;
            }
            clientsUpdateClient({clientId, client: {secret}});
            this.onPasswordPopupClose();
            fetchClientsState();
        }
    };
    onDeleteClick = () => {
        metric.goal('lk_my_app_del_click');
        this.setState({showDeletePopup: true});
    };
    onDeletePopupClose = () => this.setState({showDeletePopup: false});
    onConfirmDeleteClick = async () => {
        const {clientId, postDeleteClient, pushHistory} = this.props;
        const {status, redirectTo} = await postDeleteClient(clientId);

        if (status === 'ok') {
            if (redirectTo) {
                window.location.href = redirectTo.replace('%retpath', encodeURIComponent(window.location.href));
                return;
            }
            pushHistory(ROUTES.LIST);
        }
    };

    renderMainInfo = () => {
        const {client, clientsMap} = this.props;
        const {iconUrl, title, description, createTime, ttl} = client;

        return (
            <div className={b('mainInfo')}>
                {iconUrl && (
                    <div
                        className={b('icon')}
                        style={{backgroundImage: `url('${iconUrl.replace('/normal', '/big')}')`}}
                    />
                )}
                <div className={b('mainInfoRight')}>
                    <div className={b('title')}>{title}</div>
                    {description && <div className={b('description')}>{description}</div>}
                    {createTime && (
                        <div className={b('createTime')}>
                            {i18n('ClientInfo.info.createTime')} {dateToText(new Date(createTime * 1000))}
                        </div>
                    )}
                    <div className={b('tokenLifeTime')}>
                        {i18n('ClientInfo.info.tokenLifeTime')}{' '}
                        {getTtlInfoString({scopes: Object.values(clientsMap), ttl})}
                    </div>
                </div>
            </div>
        );
    };
    renderSubscopeList = (subScopes) =>
        Object.keys(subScopes).map((key) => (
            <div className={b('subScope')} key={subScopes[key].title}>
                •&nbsp;&nbsp;{subScopes[key].title}
                {subScopes[key].requiresApproval && (
                    <div className={b('subScopeApprovalFlag')}>
                        <ClientModerationMark />
                    </div>
                )}
            </div>
        ));
    renderScopesList = () => {
        const {client: {scopes} = {}} = this.props;

        if (!scopes) {
            return null;
        }

        return (
            <div className={b('scopesList')}>
                <div className={b('scopesTitle')}>{i18n('ClientInfo.scopes.title')}</div>
                {Object.keys(scopes).map((scopeKey) => {
                    return (
                        <div className={b('scope')} key={scopeKey}>
                            <div className={b('scopeTitle')}>{scopeKey}</div>
                            <div className={b('subScopes')}>{this.renderSubscopeList(scopes[scopeKey])}</div>
                        </div>
                    );
                })}
            </div>
        );
    };
    renderSideBlocks = () => {
        const {clientIdUrl = '', client: {secret, callback} = {}, tld = 'ru'} = this.props;

        return (
            <>
                <ClientSideInfoLayout
                    title={SIDE_INFO_TEXTS.clientId.title}
                    // eslint-disable-next-line max-len
                    href={`https://yandex.${tld}/dev/id/doc/dg/oauth/reference/auto-code-client.html#auto-code-client__get-code`}
                    description={SIDE_INFO_TEXTS.clientId.description}
                    textsToCopy={[{prefix: '', text: clientIdUrl}]}
                    goal='lk_my_app_clientID_click'
                />
                <ClientSideInfoLayout
                    title={SIDE_INFO_TEXTS.secret.title}
                    // eslint-disable-next-line max-len
                    href={`https://yandex.${tld}/dev/id/doc/dg/oauth/reference/auto-code-client.html#auto-code-client__get-code`}
                    description={SIDE_INFO_TEXTS.secret.description}
                    textsToCopy={[{prefix: '', text: secret}]}
                    goal='lk_my_app_client_secret_click'
                />
                <ClientSideInfoLayout
                    title={SIDE_INFO_TEXTS.urls.title}
                    // eslint-disable-next-line max-len
                    href={`https://yandex.${tld}/dev/id/doc/dg/oauth/reference/auto-code-client.html#auto-code-client__get-token`}
                    description={SIDE_INFO_TEXTS.urls.description}
                    textsToCopy={[{prefix: '', text: callback}]}
                    goal='lk_my_app_redirecturl_click'
                />
            </>
        );
    };
    renderButtons = () => (
        <>
            <div className={b('button')}>
                <Button width='max' view='action' size='l' onClick={this.onLoadClick}>
                    <div className={b('buttonIcon', {load: true})} />
                    {BUTTONS_TEXTS.load}
                </Button>
            </div>
            <div className={b('button')}>
                <Button width='max' view='default' size='l' onClick={this.onPasswordClick}>
                    <div className={b('buttonIcon', {password: true})} />
                    {BUTTONS_TEXTS.password}
                </Button>
            </div>
            <div className={b('button')}>
                <Button width='max' view='default' size='l' onClick={this.onEditClick}>
                    <div className={b('buttonIcon', {edit: true})} />
                    {BUTTONS_TEXTS.edit}
                </Button>
            </div>
            <div className={b('button')}>
                <Button width='max' view='default' size='l' onClick={this.onDeleteClick}>
                    <div className={b('buttonIcon', {delete: true})} />
                    {BUTTONS_TEXTS.delete}
                </Button>
            </div>
        </>
    );
    renderDeletePopup = () => {
        const {showDeletePopup} = this.state;
        const {client, isMobile, isTouch} = this.props;
        const {webCallbackUrls = [], iosAppstoreUrl = '', androidAppstoreUrl = '', turboappUrl = '', title} = client;
        const isDrawer = isMobile || isTouch;

        const Component = isDrawer ? Drawer : Modal;

        return (
            <Component
                size={isDrawer ? undefined : 'm'}
                view={isDrawer ? 'default' : undefined}
                visible={showDeletePopup}
                onClose={this.onDeletePopupClose}
            >
                <div className={b('modalContent', {isDrawer, isModal: !isDrawer})}>
                    {!isDrawer && (
                        <button type='button' className={b('modalClose')} onClick={this.onDeletePopupClose} />
                    )}
                    <div className={b('modalTitle', {isModal: !isDrawer})}>
                        {MODAL_DELETE_TEXTS.title.replace('%1', title)}
                    </div>
                    <div className={b('modalDescription')}>{MODAL_DELETE_TEXTS.description}</div>
                    <div className={b('modalPlatforms')}>
                        {webCallbackUrls.length && (
                            <div className={b('modalPlatform')}>•&nbsp;&nbsp;{MODAL_DELETE_TEXTS.web}</div>
                        )}
                        {iosAppstoreUrl && (
                            <div className={b('modalPlatform')}>•&nbsp;&nbsp;{MODAL_DELETE_TEXTS.iosAppstoreUrl}</div>
                        )}
                        {androidAppstoreUrl && (
                            <div className={b('modalPlatform')}>
                                •&nbsp;&nbsp;{MODAL_DELETE_TEXTS.androidAppstoreUrl}
                            </div>
                        )}
                        {turboappUrl && (
                            <div className={b('modalPlatform')}>•&nbsp;&nbsp;{MODAL_DELETE_TEXTS.turboappUrl}</div>
                        )}
                    </div>
                    <div className={b('modalButtons')}>
                        <div className={b('modalButton')}>
                            <Button width='max' view='default' size='l' onClick={this.onDeletePopupClose}>
                                {MODAL_BUTTONS_TEXTS.cancel}
                            </Button>
                        </div>
                        <div className={b('modalButton')}>
                            <Button width='max' view='action' size='l' onClick={this.onConfirmDeleteClick}>
                                {MODAL_BUTTONS_TEXTS.delete}
                            </Button>
                        </div>
                    </div>
                </div>
            </Component>
        );
    };
    renderPasswordPopup = () => {
        const {showPasswordPopup} = this.state;
        const {isMobile, isTouch} = this.props;
        const isDrawer = isMobile || isTouch;

        const Component = isDrawer ? Drawer : Modal;

        return (
            <Component
                size={isDrawer ? undefined : 'm'}
                view={isDrawer ? 'default' : undefined}
                visible={showPasswordPopup}
                onClose={this.onPasswordPopupClose}
            >
                <div className={b('modalContent', {isDrawer, isModal: !isDrawer})}>
                    {!isDrawer && (
                        <button type='button' className={b('modalClose')} onClick={this.onPasswordPopupClose} />
                    )}
                    <div className={b('modalTitle', {isModal: !isDrawer})}>{MODAL_PASSWORD_TEXTS.title}</div>
                    <div className={b('modalDescription')}>{MODAL_PASSWORD_TEXTS.description}</div>
                    <div className={b('modalButtons')}>
                        <div className={b('modalButton')}>
                            <Button width='max' view='default' size='l' onClick={this.onConfirmPasswordResetClick}>
                                {MODAL_BUTTONS_TEXTS.reset}
                            </Button>
                        </div>
                        <div className={b('modalButton')}>
                            <Button width='max' view='default' size='l' onClick={this.onPasswordPopupClose}>
                                {MODAL_BUTTONS_TEXTS.leave}
                            </Button>
                        </div>
                    </div>
                </div>
            </Component>
        );
    };
    render() {
        const isViewedByOwner = Boolean(this.props.client.secret);

        return (
            <div className={b()}>
                <div className={b('main')}>
                    <div className={b('mainClientInfo')}>
                        {this.renderMainInfo()}
                        {this.renderScopesList()}
                    </div>
                    {isViewedByOwner && <div className={b('mainSideBlocks')}>{this.renderSideBlocks()}</div>}
                </div>
                {isViewedByOwner && (
                    <>
                        <div className={b('buttons')}>{this.renderButtons()}</div>
                        <div className={b('stepsInfo')}>
                            <ClientNextStepsInfo />
                        </div>
                        {this.renderDeletePopup()}
                        {this.renderPasswordPopup()}
                    </>
                )}
            </div>
        );
    }
}
