import debounce from 'lodash/debounce';
import {push as pushHistory} from 'react-router-redux';
import * as editActions from '../actions/edit';
import * as clientsActions from '../actions/clients';
import {FORM_FIELDS, FORM_PLATFORMS, ROUTES} from '../const';
import {metric} from '../metric';
import {makeRequest} from './util';

const getPermissionsIdList = (scopes) =>
    Object.values(scopes).reduce((acc, permissions) => {
        acc.push(...Object.keys(permissions));

        return acc;
    }, []);

const getEditFormData = (edit = {}, csrf = '') => {
    const formData = new FormData();

    const data = Object.values(FORM_FIELDS).reduce((acc, field) => {
        if (field === FORM_FIELDS.ICON) {
            formData.set('icon', edit[field]);
        } else if (field === FORM_FIELDS.SCOPES) {
            if (edit[field]) {
                acc[field] = getPermissionsIdList(edit[field]);
            }
        } else if (field === FORM_FIELDS.ICON_URL) {
            // DO NOTHING
        } else {
            acc[field] = edit[field];
        }

        return acc;
    }, {});

    formData.set('csrf', csrf);
    if (edit.clientId) {
        formData.set('clientId', edit.clientId);
    }

    if (data.platforms) {
        if (!data.platforms[FORM_PLATFORMS.WEB]) {
            delete data[FORM_FIELDS.WEB_CALLBACK_URLS];
        }
        if (!data.platforms[FORM_PLATFORMS.ANDROID]) {
            delete data[FORM_FIELDS.ANDROID_APPSTORE_URL];
            delete data[FORM_FIELDS.ANDROID_FINGERPRINTS];
            delete data[FORM_FIELDS.ANDROID_PACKAGE_NAMES];
        }
        if (!data.platforms[FORM_PLATFORMS.IOS]) {
            delete data[FORM_FIELDS.IOS_APPSTORE_URL];
            delete data[FORM_FIELDS.IOS_APP_ID];
        }
        if (!data.platforms[FORM_PLATFORMS.TURBOAPP]) {
            delete data[FORM_FIELDS.TURBOAPP_URL];
        }
        data.platforms = Object.keys(data.platforms).filter((platform) => data.platforms[platform]);
    }

    formData.set('data', JSON.stringify(data));

    return formData;
};

export const postClientValidate = () => (dispatch, getState) => {
    const {edit = {}, common: {csrf = ''} = {}} = getState();

    const formData = getEditFormData(edit, csrf);

    return makeRequest(dispatch, '/client/api/validate', formData).then((response) => {
        dispatch(editActions.editSetValidateResult(response));
    });
};

const debouncePostClientValidateFunc = debounce((dispatch) => dispatch(postClientValidate()), 300);

export const debouncePostClientValidate = () => (dispatch) => debouncePostClientValidateFunc(dispatch);

export const fetchEditState = (clientId) => (dispatch, getState) => {
    const {common: {csrf = ''} = {}} = getState();

    const formData = new FormData();

    formData.set('csrf', csrf);
    formData.set('clientId', clientId);

    return makeRequest(dispatch, '/client/api/state/edit', formData).then((response) => {
        dispatch(editActions.editSetState(response));
        dispatch(postClientValidate());
    });
};

let scrollToError = (duration = 300) => {
    const error = document.querySelector('.error-anchor');
    const formElement = document.getElementById('form');

    const startingY = formElement.scrollTop;
    const diff = error.getBoundingClientRect().top - 100;

    let start;

    const animateScrolling = (timestamp) => {
        if (!start) {
            start = timestamp;
        }
        const time = timestamp - start;

        formElement.scrollTo(0, startingY + diff * (time / duration));
        time < duration && requestAnimationFrame(animateScrolling);
    };

    requestAnimationFrame(animateScrolling);
};

export const postClientCreate = () => (dispatch, getState) =>
    postClientValidate()(dispatch, getState).then(() => {
        const {edit = {}, common: {csrf} = {}} = getState();

        if (edit.validateResult && edit.validateResult.errors && edit.validateResult.errors.length) {
            dispatch(editActions.editSetState({showErrors: true}));
            setTimeout(() => scrollToError(300), 0);
            return; // проскролить
        }

        return makeRequest(dispatch, '/client/api/create', getEditFormData(edit, csrf)).then(
            ({client_id: clientId}) => {
                metric.goal('lk_new_clientID_success');
                dispatch(clientsActions.clientsUpdateClient({client: {...edit, clientId}, clientId}));
                dispatch(pushHistory(ROUTES.INFO.replace(':clientId', clientId)));
            }
        );
    });

export const postClientEdit = () => (dispatch, getState) =>
    postClientValidate()(dispatch, getState).then(() => {
        const {edit = {}, common: {csrf} = {}} = getState();

        if (edit.validateResult && edit.validateResult.errors && edit.validateResult.errors.length) {
            dispatch(editActions.editSetState({showErrors: true}));
            setTimeout(() => scrollToError(300), 0);
            return; // проскролить
        }

        return makeRequest(dispatch, '/client/api/edit', getEditFormData(edit, csrf)).then(() => {
            dispatch(clientsActions.clientsUpdateClient({client: edit, clientId: edit.clientId}));
            dispatch(pushHistory(ROUTES.INFO.replace(':clientId', edit.clientId)));
        });
    });

export const editFieldChange = (payload) => (dispatch) => {
    dispatch(editActions.editFieldChange(payload));
    dispatch(debouncePostClientValidate());
};

export const editFileFieldChange = (file) => (dispatch) => {
    dispatch(
        editActions.editSetState({[FORM_FIELDS.ICON]: file, [FORM_FIELDS.ICON_ID]: null, [FORM_FIELDS.ICON_URL]: null})
    );
    dispatch(debouncePostClientValidate());
};

export const editPermissionToggle = (payload) => (dispatch, getState) => {
    dispatch(editActions.editPermissionToggle(payload));
    dispatch(debouncePostClientValidate());

    const {edit: {scopes = {}} = {}, common: {permissionsById = {}} = {}} = getState();
    const permissionsId = getPermissionsIdList(scopes);
    const {ttl, requiresApproval, ttlRefreshable} = permissionsId.reduce(
        (acc, permissionId) => {
            const {ttl, turnedOn, requiresApproval, ttlRefreshable} = permissionsById[permissionId];

            acc.ttl = acc.ttl === undefined ? ttl : typeof ttl === 'number' ? Math.min(acc.ttl, ttl) : acc.ttl;
            acc.ttlRefreshable = acc.ttlRefreshable || ttlRefreshable || !ttl || ttl === 'INFINITE';
            acc.requiresApproval = acc.requiresApproval || (requiresApproval && !turnedOn);

            return acc;
        },
        {ttl: undefined, requiresApproval: false, ttlRefreshable: false}
    );

    dispatch(editActions.editSetState({ttl, ttlRefreshable, requiresApproval}));
};

export const editSetStateById = (clientId) => (dispatch, getState) => {
    const {clients: {clientsMap: {[clientId]: client} = {}} = {}} = getState();

    dispatch(client ? editActions.editSetState({...client}) : fetchEditState(clientId));
};
