import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import classnames from 'classnames';

import Button from 'lego-on-react/src/components/button/button.react';

import Spinner from './Spinner.jsx';

import {sendCode, ENTER_CODE_STATUS} from '../actions/index';
import errors from '../errors';

class EnterCodeScreen extends PureComponent {
    static mapStateToProps(state) {
        const {errors, isFetching, status, deviceCode} = state.devicePage;

        return {isFetching, status, formErrors: errors, deviceCode};
    }

    constructor(props) {
        super(props);

        this.handleSubmit = this.handleSubmit.bind(this);
        this.handleChange = this.handleChange.bind(this);
        this.state = {deviceCode: ''};
    }

    componentWillUpdate(nextProps) {
        if (this.props.status !== nextProps.status) {
            this.setState({deviceCode: nextProps.deviceCode});
            this.codeField.value = nextProps.deviceCode;
        }
    }

    handleChange() {
        const deviceCode = this.codeField.value;
        this.setState({deviceCode});
    }

    handleSubmit(event) {
        event.preventDefault();

        const deviceCode = this.codeField.value;
        this.setState({deviceCode});

        this.props.dispatch(sendCode(deviceCode.replace(/(-)?(–)?/g, '')));
    }

    render() {
        const fieldName = 'codeField';
        const {isFetching, status, formErrors} = this.props;
        const {deviceCode} = this.state;
        const hasErrors = formErrors.length !== 0;

        return (
            <div className={classnames('enter-code-screen', {
                active: status === ENTER_CODE_STATUS
            })}>
                <div className='device-code-enter-img'/>

                <h1 className='screen-title'>{i18n('frontend.device.device_connect_title')}</h1>
                <div className='screen-description'>{i18n('frontend.device.enter_device_code')}:</div>

                <form
                    className='enter-code-screen__form'
                    onSubmit={this.handleSubmit}
                    autoComplete='off'>

                    <div className={classnames('form__field', {
                        field__invalid: hasErrors,
                        field__filled: deviceCode && deviceCode.length
                    })}>
                        <span className='code-input'>
                            <input
                                id={fieldName}
                                name={fieldName}
                                ref={(field) => this.codeField = field}
                                type='text'
                                autoComplete='off'
                                placeholder={'\u2013\u2013\u2013\u2013-\u2013\u2013\u2013\u2013'}
                                onChange={this.handleChange}
                                onFocus={this.handleChange}
                                onBlur={this.handleChange}/>

                            <span/>
                        </span>

                        {hasErrors &&
                            <div className='form__field-error error-message'>
                                {errors[formErrors[0]] ? errors[formErrors[0]] : errors['form.internal']}
                            </div>
                        }
                    </div>

                    <div className='form__submit-btn'>
                        <Button
                            theme='action'
                            size='l'
                            width='max'
                            type='submit'
                            disabled={isFetching}>
                            {i18n('frontend.device.send_device_code')}
                        </Button>

                        <Spinner/>
                    </div>
                </form>
            </div>
        );
    }
}

EnterCodeScreen.propTypes = {
    deviceCode: PropTypes.string.isRequired,
    status: PropTypes.string.isRequired,
    formErrors: PropTypes.array.isRequired,
    isFetching: PropTypes.bool.isRequired
};

export default connect(EnterCodeScreen.mapStateToProps)(EnterCodeScreen);
