import React, {PureComponent} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import classnames from 'classnames';

import Button from 'lego-on-react/src/components/button/button.react';

import ClientScopesList from './ClientScopesList.jsx';
import Spinner from './Spinner.jsx';

import {
    allowClientRequest,
    changeStatus,
    setDeviceCode,
    ENTER_CODE_STATUS,
    REQUEST_PERMISSION_STATUS
} from '../actions';

class RequestPermissionScreen extends PureComponent {
    static mapStateToProps(state) {
        const {
            clientTitle,
            clientScopes,
            deviceCode,
            isFetching,
            status,
        } = state.devicePage;

        return {
            clientTitle,
            clientScopes,
            deviceCode,
            isFetching,
            status
        };
    }

    constructor(props) {
        super(props);

        this.handleAllowSubmit = this.handleAllowSubmit.bind(this);
        this.handleDenySubmit = this.handleDenySubmit.bind(this);
    }

    handleAllowSubmit(event) {
        const {dispatch, deviceCode} = this.props;

        event.preventDefault();
        dispatch(allowClientRequest(deviceCode));
    }

    handleDenySubmit(event) {
        event.preventDefault();
        this.props.dispatch(setDeviceCode(''));
        this.props.dispatch(changeStatus(ENTER_CODE_STATUS));
    }

    render() {
        const {
            clientTitle,
            clientScopes,
            isFetching,
            status
        } = this.props;

        return (
            <div className={classnames('request-permission-screen', {
                active: status === REQUEST_PERMISSION_STATUS
            })}>
                <h1 className='screen-title'>{i18n('frontend.device.approve_client_title')}</h1>
                <div className='screen-description'>
                    {i18n('frontend.authorize.authorize.request', clientTitle)}
                </div>

                <div className='authrequest-scopes'>
                    {
                        clientScopes.map((requestedScope) => (
                            <div className='clientScopesList-section' key={requestedScope.title}>
                                <div className='clientScopesList-section-title'>
                                    {requestedScope.title}
                                </div>
                                <ClientScopesList scopes={requestedScope.scopes}/>
                            </div>
                        ))
                    }
                </div>

                <div className='btn-group'>
                    <form className='deny-request-permission__form' onSubmit={this.handleDenySubmit}>
                        <div className='form__submit-btn'>
                            <Button
                                theme='normal'
                                size='l'
                                width='max'
                                type='submit'
                                disabled={isFetching}>
                                {i18n('frontend.common.common.undo')}
                            </Button>
                        </div>
                    </form>

                    <form className='allow-request-permission__form' onSubmit={this.handleAllowSubmit}>
                        <div className='form__submit-btn'>
                            <Button
                                theme='action'
                                size='l'
                                width='max'
                                type='submit'
                                disabled={isFetching}>
                                {i18n('frontend.device.approve_client')}
                            </Button>

                            <Spinner/>
                        </div>
                    </form>
                </div>

            </div>
        );
    }
}

RequestPermissionScreen.propTypes = {
    status: PropTypes.string.isRequired,
    clientTitle: PropTypes.string.isRequired,
    clientScopes: PropTypes.array.isRequired,
    deviceCode: PropTypes.string.isRequired,
    dispatch: PropTypes.func.isRequired,
    isFetching: PropTypes.bool.isRequired
};

export default connect(RequestPermissionScreen.mapStateToProps)(RequestPermissionScreen);
