const putils = require('putils');
const Plog = require('plog');
const jsesc = require('jsesc');
const he = require('he');
const utils = require('./utils');
const ClientModel = require('papi/OAuth/models/Client');
const {SERVICE_ALIASES} = require('../../lib/tvm');
const AuthorizePage = require('../../pages/authorize/AuthorizePage');

const createLogger = (req, res, next) => {
    req.logger = new Plog(req.controller.getLogId(), 'oauth', 'page', 'authorize');
    next();
};

const checkAuth = (logText) => (req, res, next) => {
    const {controller, logger} = req;

    utils.checkAuth(controller, () => {
        logger.info(logText);
        next();
    });
};

const csrfCheck = async (req, res, next) => {
    const {controller, logger} = req;

    let csrfValid;

    try {
        csrfValid = await controller.isCsrfTokenValidV2();
    } catch {
        csrfValid = false;
    }

    if (csrfValid) {
        return next();
    }

    logger.info().type('allow').write('csrf check failed, redirecting to index');
    return controller.redirectToIndex();
};

const commit = (req, res) => {
    const {controller, oauthApi, logger} = req;
    const {requestId, grantedScopes} = res.locals;
    const paymentAuthRetpath = controller.getPaymentAuthRetpath().href;

    return oauthApi.authorizeCommitV2(requestId, grantedScopes, paymentAuthRetpath).then(
        (response) => {
            const queryKeys = ['state'];

            if (response.code) {
                queryKeys.push('code');
            } else {
                queryKeys.push('access_token', 'token_type', 'expires_in');
            }

            if (response.scope) {
                queryKeys.push('scope');
            }

            const query = {};

            queryKeys.forEach((key) => {
                if (response[key]) {
                    query[key] = response[key];
                }
            });

            const redirectUrl = utils.appendQuery(response.redirect_uri, query, response.code ? '?' : '#');
            const redirectUrlJSEscaped = jsesc(redirectUrl, {isScriptContext: true});
            const redirectUrlHTMLEscaped = he.encode(redirectUrl, {strict: true});

            logger.info().type('commit').write('Redirecting to %s', response.redirect_uri);

            // https://st.yandex-team.ru/PASSP-33229
            return res.send(`
                <html>
                    <head>
                        <meta http-equiv="refresh" content="1;URL='${redirectUrlHTMLEscaped}'">
                        <script nonce="${controller._response.locals.nonce}" type="text/javascript">
                            window.location.replace('${redirectUrlJSEscaped}');
                        </script>
                    </head>
                </html>
            `);
        },
        (error) => utils.errorHandler(req, res, error)
    );
};

const render = (req, res, response) => {
    const {controller, oauthApi, logger} = req;

    return new AuthorizePage(controller, oauthApi).render(res, logger, response);
};

const renderAuthorizePage = [
    createLogger,
    checkAuth('opened /authorize'),
    (req, res, next) => {
        const {controller, serviceTickets, oauthApi} = req;
        const clientID = controller.getRequestParam('client_id');
        const responseType = controller.getRequestParam('response_type');
        const deviceId = controller.getRequestParam('device_id');
        const deviceName = controller.getRequestParam('device_name');
        const state = controller.getRequestParam('state');
        const loginHint = controller.getRequestParam('login_hint');
        const redirectUri = controller.getRequestParam('redirect_uri');
        const codeChallenge = controller.getRequestParam('code_challenge');
        const codeChallengeMethod = controller.getRequestParam('code_challenge_method');
        const requiredScopesParam = controller.getRequestParam('scope');
        const optionalScopesParam = controller.getRequestParam('optional_scope');
        const requiredScopes =
            requiredScopesParam && typeof requiredScopesParam === 'string' ? requiredScopesParam.split(' ') : undefined;
        const optionalScopes =
            optionalScopesParam && typeof requiredScopesParam === 'string' ? optionalScopesParam.split(' ') : undefined;

        if (!clientID) {
            return utils.showRequestError(req, 'authorize.errors.client_id_missing', 'Client id is not defined');
        }

        if (!ClientModel.isValidClientId(clientID)) {
            return utils.showRequestError(req, 'authorize.errors.client_id_unknown', 'Client id is invalid');
        }

        if (loginHint) {
            const headers = {
                'X-Ya-Service-Ticket': serviceTickets && serviceTickets[SERVICE_ALIASES.BLACKBOX]
            };

            return controller
                .getAuth()
                .userInfo({login: loginHint}, headers)
                .then((user) => {
                    const {id: uid} = user || {};

                    if (!uid) {
                        return oauthApi
                            .authorizeSubmitV2(
                                clientID,
                                responseType,
                                redirectUri,
                                state,
                                deviceId,
                                deviceName,
                                requiredScopes,
                                optionalScopes,
                                codeChallenge,
                                codeChallengeMethod
                            )
                            .then(
                                (response) => utils.loginHintErrorHandler(req, user, response.redirect_uri),
                                (error) => {
                                    const redirectUri = error.getResponse().redirect_uri;

                                    return redirectUri
                                        ? utils.loginHintErrorHandler(req, user, redirectUri)
                                        : utils.errorHandler(req, res, error);
                                }
                            );
                    }

                    if (uid !== controller.getAuth().getUid()) {
                        const authUrl = new URL(controller.getAuthUrl().href);

                        let retpath = new URL(controller.getUrl().href);

                        retpath.searchParams.delete('login_hint');

                        if (controller.isIntranet()) {
                            retpath = new URL(controller.getAuthUrl().href);
                            retpath.search = new URLSearchParams({
                                mode: 'add-user',
                                login: loginHint,
                                retpath: retpath.toString()
                            }).toString();
                        }

                        authUrl.pathname = 'passport';
                        authUrl.search = new URLSearchParams({
                            mode: 'embeddedauth',
                            origin: controller.getRequestParam('origin') || 'oauth',
                            action: 'change_default',
                            uid,
                            login: loginHint,
                            yu: controller.getCookie('yandexuid'),
                            retpath: retpath.toString()
                        }).toString();

                        return controller.redirect(authUrl.toString());
                    }

                    next();
                })
                .catch((error) => utils.errorHandler(req, res, error));
        }

        next();
    },
    (req, res, next) => {
        const {controller, oauthApi, logger} = req;
        const clientID = controller.getRequestParam('client_id');
        const responseType = controller.getRequestParam('response_type');
        const codeChallenge = controller.getRequestParam('code_challenge');
        const codeChallengeMethod = controller.getRequestParam('code_challenge_method');
        const paymentAuthScheme = controller.getRequestParam('payment_auth_scheme');
        const requiredScopesParam = controller.getRequestParam('scope');
        const optionalScopesParam = controller.getRequestParam('optional_scope');
        const requiredScopes =
            requiredScopesParam && typeof requiredScopesParam === 'string' ? requiredScopesParam.split(' ') : undefined;
        const optionalScopes =
            optionalScopesParam && typeof optionalScopesParam === 'string' ? optionalScopesParam.split(' ') : undefined;

        return oauthApi
            .authorizeSubmitV2(
                clientID,
                responseType,
                controller.getRequestParam('redirect_uri'),
                controller.getRequestParam('state'),
                controller.getRequestParam('device_id'),
                controller.getRequestParam('device_name'),
                requiredScopes,
                optionalScopes,
                codeChallenge,
                codeChallengeMethod,
                paymentAuthScheme
            )
            .then(
                (response) => {
                    res.locals = {...res.locals, requestId: response.request_id, grantedScopes: []};

                    const isConfirmRequired =
                        response.require_user_confirm ||
                        ['yes', 'true', '1'].includes(controller.getRequestParam('force_confirm'));

                    if (isConfirmRequired) {
                        logger.info().type('submit').write('Authorization request requires user confirmation');

                        return render(req, res, response);
                    }

                    logger.info().type('submit').write('Authorization request granted immediately');
                    next();
                },
                (error) => utils.errorHandler(req, res, error)
            );
    },
    commit
];

const handleAuthorizeFormPost = [
    createLogger,
    checkAuth('opened /authorize/allow'),
    csrfCheck,
    (req, res, next) => {
        const {controller, oauthApi} = req;
        const requestId = controller.getRequestParam('request_id');

        let grantedScopes = controller.getRequestParam('granted_scopes');

        if (typeof grantedScopes === 'string') {
            grantedScopes = [grantedScopes];
        }

        res.locals = {...res.locals, requestId, grantedScopes};

        if (Array.isArray(grantedScopes) && grantedScopes.length) {
            return next();
        }

        return oauthApi.authorizeGetStateV2(requestId).then(
            ({redirect_uri: redirectUri}) => {
                return utils.errorRedirect(
                    req,
                    redirectUri,
                    'invalid_scope',
                    putils.i18n(controller.getLang(), 'authorize.errors.scopes_empty')
                );
            },
            (error) => utils.errorHandler(req, res, error)
        );
    },
    commit
];

const handleAuthorizeCommitGet = [
    createLogger,
    checkAuth('opened /authorize/commit'),
    (req, res, next) => {
        const {controller} = req;
        const requestId = controller.getRequestParam('passportRequestId');
        const grantedScopes = [];

        if (!requestId) {
            return utils.showRequestError(req, 'authorize.errors.request.not_found', 'Request id is not defined');
        }

        res.locals = {...res.locals, requestId, grantedScopes};
        return next();
    },
    commit
];

const handleAuthorizeFingerprint = [
    createLogger,
    checkAuth('send fingerprint /authorize/logger'),
    csrfCheck,
    (req, res) => {
        res.send('');

        const info = utils.getStatboxInfo(req);

        if (info) {
            req.logger.info().type('/authorize fingerprint').write('send fingerprint to statbox');

            return req.passportApi.writeStatbox(info);
        } else {
            req.logger.warn().type('/authorize fingerprint').write('no fingerprint');
        }
    }
];

module.exports = (app) => {
    app.get('/authorize', renderAuthorizePage);
    app.get('/authorize/commit', handleAuthorizeCommitGet);

    app.post('/authorize/allow', handleAuthorizeFormPost);
    app.post('/authorize/logger', handleAuthorizeFingerprint);
};
