const querystring = require('querystring');
const ApiError = require('papi/OAuth').ApiError;
const putils = require('putils');
const {getRedirectErrors, getImmediateExceptions} = require('./const');
const ErrorPage = require('../../pages/error/ErrorPage');
const OutboundRedirectPage = require('../../pages/authorize/outboundRedirect/OutboundRedirectPage');
const HTTPException = require('../../lib/exceptions/http/HTTPException');

const trimSlashesRegexp = /(^\/|\/$)/g;
const spacesRegexp = /\s+/g;

const showRequestError = (req, messageKey, reason) => {
    const {logger, controller, oauthApi} = req;

    logger.info().type('submit').write(reason);
    return new ErrorPage(controller, oauthApi)
        .setError(new HTTPException(400, putils.i18n(controller.getLang(), messageKey)))
        .open();
};

const isYandexUrl = (yandexURL) => {
    const host = new URL(yandexURL).hostname;

    return /(:?^|.+\.)yandex\.(:?ru|com|com\.tr|uk|kz|be)$/.test(host);
};

const appendQuery = (url, query, appendAfterSymbol) => {
    appendAfterSymbol = appendAfterSymbol || '?';

    const afterSymbol = !url.includes(appendAfterSymbol) ? appendAfterSymbol : '&';

    const queryStr = querystring.stringify(
        Object.keys(query).reduce((acc, key) => {
            if (query[key]) {
                acc[key] = query[key];
            }

            return acc;
        }, {})
    );

    return url + afterSymbol + queryStr;
};

const checkAuth = (controller, cb) => {
    if (!controller.getAuth().isLoggedIn()) {
        const login = controller.getRequestParam('login_hint');
        const retpath = controller.getRequestParam('retpath');
        const queryParams = {
            noreturn: 1,
            origin: controller.getRequestParam('origin') || 'oauth'
        };

        if (login) {
            queryParams.login = login;
        }

        if (retpath) {
            queryParams.retpath = retpath;
        }

        return controller.getAuth().authorize(queryParams);
    }

    if (controller.getAuth().isAutologged()) {
        return controller.getAuth().lightauth2full(true);
    }

    return cb();
};

const errorRedirect = (req, redirectUrl, error, message, immediate, isErrorInQuery = true) => {
    const {logger, controller, oauthApi} = req;
    const responseType = controller.getRequestParam('response_type');
    const state = controller.getRequestParam('state');
    const queryParams = {state};

    if (isErrorInQuery) {
        Object.assign(queryParams, {error, error_description: message});
    }
    redirectUrl = appendQuery(redirectUrl, queryParams, responseType === 'token' ? '#' : '?');

    if (immediate) {
        logger.info().type('errorRedirect').write('Immediate redirect to %s', redirectUrl);
        return controller.redirect(redirectUrl);
    }

    if (isYandexUrl(redirectUrl)) {
        logger.info().type('errorRedirect').write('Redirecting to yandex url %s', redirectUrl);
        return controller.redirect(redirectUrl);
    }

    const outboundRedirect = new OutboundRedirectPage(controller, oauthApi);

    if (message) {
        outboundRedirect.errorDescription(message);
    }

    logger.info().type('errorRedirect').write('Outbound redirect to %s, asking for confirmation', redirectUrl);
    return outboundRedirect.url(redirectUrl).errorCode(error).open();
};

const errorHandler = (req, res, error) => {
    const {controller, logger, oauthApi} = req;

    if (!(error instanceof ApiError)) {
        logger.error().type('errorHandler').write('Not an api error gets rethrown:', error);
        throw error;
    }

    const immediateExceptions = getImmediateExceptions(controller);
    const matchedImmediateExceptionCode = Object.keys(immediateExceptions).find((code) => error.contains(code));

    if (matchedImmediateExceptionCode) {
        const matchedImmediateException = immediateExceptions[matchedImmediateExceptionCode];

        logger
            .info()
            .type('errorHandler')
            .write(
                'Error encountered we can not notify the client about:',
                error,
                'responding with',
                matchedImmediateException
            );

        return new ErrorPage(controller, oauthApi).setError(matchedImmediateException).open();
    }

    const redirectErrors = getRedirectErrors(error, controller);

    const matchedRedirectErrorCode = Object.keys(redirectErrors).find((code) => error.contains(code));

    if (matchedRedirectErrorCode) {
        const matchedRedirectError = redirectErrors[matchedRedirectErrorCode];

        matchedRedirectError.description = putils.i18n(controller.getLang(), matchedRedirectError.description);
        if (matchedRedirectError.redirect) {
            error.getResponse().redirect_uri = matchedRedirectError.redirect_uri;
        }

        const redirectUri = error.getResponse().redirect_uri;

        if (matchedRedirectError.reload_page) {
            const requestParams = {
                response_type: controller.getRequestParam('response_type'),
                client_id: controller.getRequestParam('client_id'),
                device_id: controller.getRequestParam('device_id'),
                device_name: controller.getRequestParam('device_name'),
                display: controller.getRequestParam('display'),
                login_hint: controller.getRequestParam('login_hint'),
                scope: controller.getRequestParam('scope'),
                optional_scope: controller.getRequestParam('optional_scope'),
                state: controller.getRequestParam('state'),
                redirect_uri: controller.getRequestParam('redirect_uri'),
                retpath: controller.getRequestParam('retpath')
            };
            const forceConfirm = ['yes', 'true', '1'].includes(controller.getRequestParam('force_confirm'));

            if (!forceConfirm) {
                requestParams.force_confirm = 'yes';
            }

            return controller.redirect(appendQuery('/authorize', requestParams));
        }

        return redirectUri
            ? errorRedirect(
                  req,
                  redirectUri,
                  matchedRedirectError.code,
                  matchedRedirectError.description,
                  false,
                  !matchedRedirectError.isNoErrorInQuery
              )
            : showRequestError(req, 'authorize.errors.request.not_found', 'There now redirect uri');
    }

    logger.error().type('errorHandler').write('Unexpected ApiError', error);
    throw error; // Re-throw an unexpected error
};

const loginHintErrorHandler = (req, user, redirectUri) => {
    const {controller} = req;
    const errorCode = user ? 'invalid_request' : 'temporarily_unavailable';
    const errorDescription = putils.i18n(
        controller.getLang(),
        `authorize.errors.login_hint.${user ? 'invalid' : 'request_error'}`
    );

    return errorRedirect(req, redirectUri, errorCode, errorDescription);
};

const statboxUtils = {
    tryToGetLog: function (json, secret) {
        let log;
        const encrypted = statboxUtils.checkIfEncrypted(json);

        try {
            if (encrypted) {
                json = putils.simpleCipher.decode(secret, json);
            }
            log = JSON.parse(json);
        } catch (e) {
            return null;
        }

        if (
            log &&
            (!log.withTrackId || log.track_id) &&
            typeof log.url === 'string' &&
            log.url &&
            typeof log.action === 'string' &&
            log.action
        ) {
            return log;
        } else {
            return null;
        }
    },
    checkIfEncrypted: (data) => Boolean(data && data.indexOf('{') === -1),
    normalizeUrl: (fullUrl) => {
        const pathname = new URL(fullUrl).pathname;

        return typeof pathname === 'string' ? `/${pathname.replace(trimSlashesRegexp, '')}` : '';
    }
};

const getStatboxInfo = function (req) {
    const data = req.body.log;
    const trackId = req.body.track_id;
    const yandexuid = req.cookies.yandexuid;
    const log = statboxUtils.tryToGetLog(data, trackId);

    if (log) {
        const info = Object.assign({}, log, {
            ignoreMissedTrack: true,
            ip: req.headers['x-real-ip'],
            user_agent: req.headers['user-agent'],
            host: req.hostname,
            yandexuid,
            url: statboxUtils.normalizeUrl(log.url) || 'unknown',
            action: log.action.replace(spacesRegexp, '_').toLowerCase()
        });

        delete info.withTrackId;
        delete info.token;
        delete info.ignoreMissedTrack;
        delete info.track_id;

        return info;
    }
};

module.exports = {
    appendQuery,
    errorHandler,
    errorRedirect,
    checkAuth,
    showRequestError,
    loginHintErrorHandler,
    getStatboxInfo
};
