const ClientPage = require('../../pages/client/new/ClientPage');
const {mapClient, mapClientsList} = require('../../pages/client/new/utils');
const {SERVICE_ALIASES, serviceTicketsSetup} = require('../../lib/tvm');
const {checkAuth, checkCsrf, handleErrors} = require('./utils');

const renderClientPage = (req) => {
    const {controller, oauthApi} = req;

    return new ClientPage(controller, oauthApi).open(
        req,
        req.serviceTickets && req.serviceTickets[SERVICE_ALIASES.BLACKBOX]
    );
};

const getFormArgs = (req, res, next) => {
    const {logger, body: {data, icon, clientId} = {}} = req;

    let formData;

    try {
        formData = JSON.parse(data);
    } catch (e) {
        logger.error('JSON parsing error');
        return res.json({status: 'error'});
    }

    const formArgs = {
        clientId,
        title: formData.title,
        description: formData.description,
        homepage: formData.homepage,
        isYandex: formData.isYandex,
        iconId: formData.iconId,
        iconFile: icon,
        platforms: formData.platforms,
        contact: formData.contact,
        iosAppId: formData.iosAppIds,
        iosAppstoreUrl: formData.iosAppstoreUrl,
        androidPackageName: formData.androidPackageNames,
        androidAppstoreUrl: formData.androidAppstoreUrl,
        androidCertFingerprints: formData.androidFingerprints,
        redirectUris: formData.webCallbackUrls,
        turboappBaseUrl: formData.turboappUrl,
        scopes: formData.scopes
    };

    res.locals.formArgs = formArgs;
    res.locals.formData = formData;

    next();
};

const handleValidateClient = async (req, res) => {
    const {logger, oauthApi, params: {clientId} = {}} = req;
    const {formData, formArgs} = res.locals;

    logger.info('Validating client %s with %j', clientId, formData);

    return oauthApi
        .validateClientChangesV3(formArgs, {requirePlatform: true})
        .then((response) => {
            logger.info('Client %s validated successfully', response.client_id);

            res.json({
                invalidateTokens: response.invalidate_tokens,
                tokenLifeSpan: response.token_ttl,
                shouldBeModerated: response.will_require_approval
            });
        })
        .catch(handleErrors(req, res));
};

const handleCreateClient = async (req, res) => {
    const {logger, oauthApi} = req;
    const {formData, formArgs} = res.locals;

    logger.info('Creating new client with %j', formData);

    return oauthApi
        .createClientV3(formArgs, {requirePlatform: true})
        .then((response) => {
            logger.info('Client %s saved successfully', response.client_id);
            return res.json(response);
        })
        .catch(handleErrors(req, res));
};

const handleEditClient = async (req, res) => {
    const {logger, oauthApi, params: {clientId} = {}} = req;
    const {formData, formArgs} = res.locals;

    logger.info('Saving client %s with %j', clientId, formData);

    return oauthApi
        .editClientV3(formArgs, {requirePlatform: true})
        .then((response) => {
            logger.info('Client %s saved successfully', response.client_id);
            return res.json(response);
        })
        .catch(handleErrors(req, res));
};

const handleClientsState = async (req, res) => {
    const {oauthApi} = req;
    const listClients = await oauthApi.listCreatedClientsVer2New();

    res.json(mapClientsList(listClients));
};

const handleEditState = async (req, res) => {
    const {oauthApi, body: {clientId} = {}} = req;
    const {client = {}, token_ttl = null} = await oauthApi.clientInfoVer3New(clientId, true);

    client.token_ttl = token_ttl;

    res.json(mapClient(client));
};

const handleDelete = async (req, res) => {
    const {logger, oauthApi, body: {clientId} = {}} = req;

    logger.info('Deleting client %s', clientId);

    return oauthApi
        .deleteClientV2(clientId)
        .then((response) => {
            logger.info('Client was delete %s successfully', response.client_id);
            return res.json(response);
        })
        .catch(handleErrors(req, res));
};

const handlePasswordReset = async (req, res) => {
    const {logger, oauthApi, body: {clientId} = {}} = req;

    logger.info('Resetting password client %s', clientId);

    return oauthApi
        .newClientSecretV2(clientId)
        .then((response) => {
            logger.info('Client was reset password %s successfully', response.client_id);
            return res.json(response);
        })
        .catch(handleErrors(req, res));
};

module.exports = (app) => {
    app.get('/', checkAuth, serviceTicketsSetup, renderClientPage);
    app.get('/client/my', (req) => req.controller.redirectToIndex());
    app.get('/client/new', checkAuth, serviceTicketsSetup, renderClientPage);
    app.get('/client/new/:tag', checkAuth, serviceTicketsSetup, renderClientPage);
    app.get('/client/edit/:clientId', checkAuth, serviceTicketsSetup, renderClientPage);
    app.get('/client/:clientId', checkAuth, serviceTicketsSetup, renderClientPage);

    app.post('/client/api/create', checkAuth, checkCsrf, getFormArgs, handleCreateClient);
    app.post('/client/api/edit', checkAuth, checkCsrf, getFormArgs, handleEditClient);
    app.post('/client/api/validate', checkAuth, checkCsrf, getFormArgs, handleValidateClient);
    app.post('/client/api/state/clients', checkAuth, checkCsrf, handleClientsState);
    app.post('/client/api/state/edit', checkAuth, checkCsrf, handleEditState);
    app.post('/client/api/delete', checkAuth, checkCsrf, handleDelete);
    app.post('/client/api/password/reset', checkAuth, checkCsrf, handlePasswordReset);
};
