const putils = require('putils');
const OauthApi = require('papi/OAuth');
const Page404 = require('../../pages/error/Error404Page');
const HTTPException = require('../../lib/exceptions/http/HTTPException');
const ErrorPage = require('../../pages/error/ErrorPage');

module.exports.checkAuth = (req, res, next) => {
    const {controller, logger} = req;

    if (!controller.getAuth().isLoggedIn()) {
        logger.info('User not logged in, redirecting to auth');
        return controller.getAuth().authorize({noreturn: 1});
    }

    if (controller.getAuth().isAutologged()) {
        logger.info('User logged in with autologin, redirecting to lightauth2full');
        return controller.getAuth().lightauth2full(true);
    }

    next();
};

module.exports.checkCsrf = async (req, res, next) => {
    const {controller, logger} = req;

    const csrf = controller.getRequestParam('csrf');

    if (!csrf) {
        logger.error('CSRF-token is required');
        return res.json({status: 'error', errors: ['form.internal']});
    }

    let csrfValid;

    try {
        csrfValid = await controller.isCsrfTokenValidV2();
    } catch {
        csrfValid = false;
    }

    if (!csrfValid) {
        logger.error('CSRF-token is not valid');
        return controller._response.json({status: 'error', errors: ['form.internal']});
    }

    next();
};

module.exports.handleErrors = (req, res) => (error) => {
    const {controller, oauthApi, logger} = req;

    if (error instanceof OauthApi.ApiError) {
        if (error.contains('sslsession.required') || error.contains('password.required')) {
            return req.method === 'GET'
                ? controller.redirectToVerifyPage(controller.getUrl().href)
                : res.json({status: 'ok', redirectTo: controller.getAuthVerifyUrl('%retpath')});
        }
        if (error.contains('client.not_found')) {
            return new Page404(controller, oauthApi).open();
        }
        if (
            error.contains('client.creator_required') ||
            error.contains('client.not_editable') ||
            error.contains('common.errors.access_denied')
        ) {
            return new ErrorPage(controller, oauthApi)
                .setError(new HTTPException(403, putils.i18n(controller.getLang(), 'common.errors.access_denied')))
                .open();
        }

        return res.json(error._response);
    }

    logger.info('Not an API error, gets rethrown: %s', error);
    throw error;
};
