const path = require('path');
const I18n = require('./i18n-webpack');
const locs = require('../loc/locs');
const webpack = require('webpack');
const MiniCssExtractPlugin = require('mini-css-extract-plugin');
const TerserJSPlugin = require('terser-webpack-plugin');
const OptimizeCSSAssetsPlugin = require('optimize-css-assets-webpack-plugin');
const args = require('yargs').argv;
const resolvePaths = require('./webpack.paths');
const SERVER_BUILD_PATH = path.resolve(__dirname, '../templates/');
const CLIENT_BUILD_PATH = path.resolve(__dirname, '../public/js');

const entry = args.page;
const langs = args.langs.split(' ');
const type = args.type;
const ENV = args.env || 'production';
const output = entry.replace('/', '.');

const configs = [];

langs.forEach((language) => {
    const clientConfig = {
        name: 'clientside',
        entry: [
            'babel-polyfill',
            'core-js/features/array/find',
            'core-js/features/array/find-index',
            'core-js/features/array/for-each',
            'core-js/features/array/includes',
            'core-js/features/array/map',
            'core-js/features/array/reduce',
            'core-js/features/object/entries',
            'core-js/features/object/assign',
            'core-js/features/string/includes',
            'core-js/features/number/is-finite',
            'core-js/features/string/starts-with',
            'core-js/es/map',
            'core-js/es/set',
            'core-js/es/promise',
            'isomorphic-fetch',
            'raf/polyfill',
            path.resolve('./react-pages', entry, 'client.jsx')
        ],
        target: 'web',
        mode: ENV,
        devtool: ENV === 'development' ? '#inline-source-map' : false,
        context: path.resolve(__dirname, '..'),
        output: {
            path: CLIENT_BUILD_PATH,
            filename: `${output}.react.${language}.js`
        },
        externals: {
            react: 'React',
            'react-dom': 'ReactDOM'
        },
        optimization: {
            minimizer: []
        },
        module: {
            rules: [
                {
                    include: /node_modules/,
                    use: 'base64-inline-loader',
                    test: /\.(jpe?g|png|ttf|gif|eot|svg|woff(2)?)(\?[a-z0-9=&.]+)?$/
                },
                {
                    test: /\.jsx?$/,
                    use: [
                        {
                            loader: 'babel-loader',
                            options: {
                                cacheDirectory: true,
                                caller: {target: 'clientside', name: 'babel-loader'}
                            }
                        }
                    ],
                    exclude: /(plibs\/greed)/
                },
                {
                    test: /\.styl$/,
                    use: [
                        MiniCssExtractPlugin.loader,
                        {
                            loader: 'css-loader',
                            options: {
                                sourceMap: true,
                                url: false
                            }
                        },
                        {
                            loader: 'postcss-loader',
                            options: {sourceMap: true}
                        },
                        {
                            loader: 'stylus-loader',
                            options: {sourceMap: true}
                        }
                    ]
                },
                {
                    test: /\.css$/,
                    include: /node_modules/,
                    use: [
                        MiniCssExtractPlugin.loader,
                        {
                            loader: 'css-loader',
                            options: {sourceMap: true}
                        },
                        {
                            loader: 'postcss-loader',
                            options: {sourceMap: true}
                        }
                    ]
                }
            ],
            noParse: [/plibs\/greed\/greed/]
        },
        resolve: resolvePaths,
        plugins: [
            new webpack.DefinePlugin({
                'process.env.BROWSER': true
            }),
            new MiniCssExtractPlugin({
                filename: `../css/${entry.replace(/[/]/g, '.')}.react.css`
            }),
            new I18n(locs[language], null, language)
        ],
        node: {
            console: ENV === 'development',
            global: true,
            process: true,
            Buffer: false,
            __filename: false,
            __dirname: false,
            setImmediate: true
        }
    };

    if (entry === 'device') {
        clientConfig.entry.push('es5-shim');
        clientConfig.entry.push('es5-shim/es5-sham');
    }

    const serverConfig = {
        devtool: false,
        name: 'serverside',
        entry: path.resolve('./react-pages', entry, 'server.jsx'),
        target: 'node',
        mode: ENV,
        output: {
            path: SERVER_BUILD_PATH,
            filename: `${output}.react.${language}.js`,
            libraryTarget: 'commonjs2'
        },
        module: {
            rules: [
                {
                    test: /\.js$/,
                    include: /pclientjs|broker|Metrika|plibs\/greed/,
                    loader: 'null-loader'
                },
                {
                    test: /\.jsx?$/,
                    use: [
                        {
                            loader: 'babel-loader',
                            options: {
                                cacheDirectory: true,
                                caller: {target: 'serverside', name: 'babel-loader'}
                            }
                        }
                    ]
                },
                {
                    test: /\.styl$/,
                    loader: 'null-loader'
                },
                {
                    test: /\.css$/,
                    loader: 'null-loader'
                }
            ]
        },
        resolve: resolvePaths,
        plugins: [
            new webpack.DefinePlugin({
                'process.env.BROWSER': false,
                'process.env.NODE_ENV': ENV === 'development' ? '"development"' : '"production"'
            }),
            new I18n(locs[language], null, language)
        ]
    };

    if (ENV !== 'development') {
        clientConfig.optimization.minimizer.push(
            new TerserJSPlugin({parallel: false, extractComments: false, sourceMap: true}),
            new OptimizeCSSAssetsPlugin({})
        );
    }

    configs.push(type === 'clientside' ? clientConfig : serverConfig);
});

module.exports = configs;
