const url = require('url');
const {Validator} = require('express-json-validator-middleware');
const DollarConfig = require('dollar-config');
const api = require('./api.js');
const validator = new Validator({allErrors: true}); // Pass in options to the Ajv instance
const {validate} = validator;
const querySchema = require('./query.schema.json');
const SsoError = require('../../error/SsoError.js');
const config = new DollarConfig(require('../../../configs/current/main.json'));
const statbox = require('../../log/helper.js');
const getCSPHeader = require('../csp/index.js');

const {parse: urlParse, format: urlFormat} = url;

const getDomain = urlString => url.parse(urlString).hostname.split('.').slice(-2).join('.');
const checkReferrer = fieldName => (req, res, next) => {
    if (fieldName === 'retpath') {
        return next();
    }

    const origin = url.parse(req.headers.referer).hostname.split('.').slice(-2).join('.');
    const queryOrigin = url.parse(req.query[fieldName]).hostname.split('.').slice(-2).join('.');

    if (queryOrigin === origin) {
        return next();
    }

    throw new SsoError('invalid_origin');
};

const getBuildMiddlewareGroup = function (opts) {
    const {
        isBackground,
        handle,
        hostFieldName
    } = opts;

    return [
        validate({query: querySchema}),
        checkReferrer(hostFieldName),
        getCSPHeader(hostFieldName),
        (req, res, next) => {
            const {
                headers,
                query: {
                    uuid,
                    [hostFieldName]: origin
                }
            } = req;

            const targetHost = urlParse(origin).hostname;
            const root = req.hostname;

            return api({
                headers,
                uuid,
                isBackground,
                targetHost
            }).then(result => {
                const {
                    container,
                    cookies,
                    headers,
                    current_domain: currentDomain,
                    cookie_check_value: cookieCheckValue,
                    auth_status: authStatus
                } = result.body;

                const {
                    scriptNonce
                } = res.locals;

                if (cookies) {
                    res.append('Set-Cookie', cookies);
                }

                if (headers) {
                    Object.keys(headers).forEach(field => {
                        if (field === 'Content-Security-Policy') {
                            return;
                        }

                        res.append(field, headers[field]);
                    });
                }

                const domain = getDomain(origin);
                const hostSsoPrefix =
                    config.get(`hostsByOrigin.${domain}.sso`) ||
                    config.get('hosts.sso');

                const context = {
                    host: urlFormat({
                        protocol: 'https',
                        hostname: `${hostSsoPrefix}${domain}`,
                        pathname: handle,
                        query: {
                            uuid
                        }
                    }),
                    [hostFieldName]: origin,
                    targetHost: getDomain(origin),
                    container,
                    scriptNonce,
                    uuid,
                    authStatus,
                    cookieCheckValue,
                    currentDomain,
                    expiresTimeout: config.get('timeouts.noauth'),
                    expiresBlockedTimeout: config.get('timeouts.blocked'),
                    root
                };

                res.locals.context = context;

                return next();
            }).catch(error => {
                statbox({
                    uuid,
                    origin: root,
                    target: origin,
                    headers: req.headers,
                    cookies: req.cookies,
                    action: 'build_container',
                    result: error.code,
                    event: (isBackground && 'pull') || 'push',
                    status: 'error'
                });

                return next(error);
            });
        }
    ];
};

module.exports = getBuildMiddlewareGroup;
