const url = require("url");
const { Validator } = require("express-json-validator-middleware");
const DollarConfig = require("dollar-config");
const config = new DollarConfig(require("../../../configs/current/main.json"));
const statbox = require("../../log/helper.js");
const querySchema = require("./query.schema.json");
const bodySchema = require("./body.schema.json");
const api = require("./api.js");
const getCSPHeader = require("../csp/index.js");

const validator = new Validator({ allErrors: true }); // Pass in options to the Ajv instance
const { validate } = validator;
const { parse: urlParse, format: urlFormat } = url;
const getDomain = (urlString) => urlString.split(".").slice(-2).join(".");

const getUseMiddlewareGroup = function (opts) {
    const { isBackground } = opts;

    return [
        validate({
            query: querySchema,
            body: bodySchema,
        }),
        getCSPHeader("retpath"),
        (req, res, next) => {
            const {
                query: { uuid },
                body: { container, retpath },
                headers,
            } = req;

            return api({
                container,
                headers,
                isBackground,
                retpath,
                uuid,
            })
                .then((result) => {
                    const {
                        auth_cookie_valid: authCookieValid,
                        cookie_check_value: cookieCheckValue,
                        cookies,
                        current_domain: currentDomain,
                        headers,
                        origin,
                        retpath,
                    } = result.body;
                    const { scriptNonce } = res.locals;
                    const root = `${config.get("hosts.leader")}${origin}`;
                    const retpathObj = urlParse(retpath, true);
                    const retpathWithError = urlFormat(
                        Object.assign({}, retpathObj, {
                            query: Object.assign({}, retpathObj.query, {
                                sso_failed: "error",
                                uuid,
                            }),
                            search: null,
                        })
                    );
                    const retpathWithBlocked = urlFormat(
                        Object.assign({}, retpathObj, {
                            query: Object.assign({}, retpathObj.query, {
                                sso_failed: "blocked",
                                uuid,
                            }),
                            search: null,
                        })
                    );

                    const expiresTimeout =
                        (authCookieValid && config.get("timeouts.auth")) ||
                        config.get("timeouts.noauth");
                    const authStatus = (authCookieValid && "auth") || "noauth";

                    if (headers) {
                        Object.keys(headers).forEach((field) => {
                            if (field === "Content-Security-Policy") {
                                return;
                            }

                            res.append(field, headers[field]);
                        });
                    }

                    if (cookies) {
                        res.append("Set-Cookie", cookies);
                    }

                    const context = {
                        authCookieValid,
                        authStatus,
                        cookieCheckValue,
                        currentDomain,
                        expiresTimeout,
                        expiresBlockedTimeout: config.get("timeouts.blocked"),
                        retpathWithBlocked,
                        retpathWithError,
                        retpath: urlFormat(retpathObj),
                        root,
                        scriptNonce,
                        uuid,
                    };

                    res.locals.context = context;
                    return next();
                })
                .catch((error) => {
                    statbox({
                        uuid,
                        origin: getDomain(req.hostname),
                        target: urlParse(retpath).hostname,
                        headers: req.headers,
                        cookies: req.cookies,
                        action: "use_container",
                        result: error.code,
                        event: (isBackground && "pull") || "push",
                        status: "error",
                    });

                    return next(error);
                });
        },
    ];
};

module.exports = getUseMiddlewareGroup;
