(function (window, document) {
    var MDA2 = window.MDA2 || (window.MDA2 = (function () {
        var _guardCookieName = 'sso_uuid';
        var _statusCookieName = 'sso_status';
        var _beaconCookieName = 'mda2_beacon';
        var _timeout = null;
        var _timeoutBeacon = null;
        var _uuid;
        var _root;
        var _sso;
        var _listener;
        var _originSsoHost;
        var _sourceHost;
        var _source;
        var _sourceBeacon;
        var _beaconRoot;
        var _beaconTimeoutTime = 50;
        var SYNC = 'synchronized';
        var SYNC_NO_BEACON = 'synchronized_no_beacon';
        var frames = {
            _iframe: null,
            _iframeBeacon: null
        };

        return {
            init: init,
            sync: sync
        };

        function init(opts) {
            _uuid = opts.uuid;
            _root = opts.root;
            _sso = opts.sso;
            _listener = opts.listener;
            _originSsoHost = 'https://' + _sso;
            _sourceHost = 'https://' + _root;
            _source = _sourceHost + '/pull?origin=' + encodeURIComponent(window.location.href) + '&uuid=' + encodeURIComponent(_uuid);
            _sourceBeacon = _sourceHost + '/beacon';

            if (window.addEventListener) {
                window.addEventListener('message', onMessage);
            } else {
                window.attachEvent('onmessage', onMessage);
            }
        }

        function sync() {
            var lastStatus = getStatus();
            var processUuid = _getCookie(_guardCookieName);
            var beaconSelf = _getCookie(_beaconCookieName);
            var maxTime = 25000;
            var expires = new Date((new Date()).getTime() + maxTime);
            var result = 'timeout';

            var iframe;

            if (processUuid) {
                _listener({
                    uuid: processUuid,
                    status: 'inprogress'
                });

                return;
            }

            if (lastStatus !== null) {
                result = 'beacon';

                // Мы знаем, что маячок недоступен
                if (lastStatus === SYNC_NO_BEACON) {
                    _beaconRoot = 0;
                }

                // Статус маячка неизвестен
                if (typeof _beaconRoot === 'undefined') {

                    // Пытаемся получить маячок с увеличивающимися интервалами
                    if (_beaconTimeoutTime < 25000) {
                        window.clearTimeout(_timeoutBeacon);
                        _beaconTimeoutTime *= 2;
                        _timeoutBeacon = window.setTimeout(sync, _beaconTimeoutTime);

                        if (!frames._iframeBeacon) {
                            setupIframe({
                                _iframe: '_iframeBeacon',
                                _source: _sourceBeacon
                            });
                        }

                        return;
                    }

                    // Перестаем пытаться по таймауту
                    _beaconRoot = -1;
                }

                // Нет доступа к маячку из-за блокировки 3rd-party, пришло
                // Number(''), cейчас или при прошлой попытке
                if (_beaconRoot === 0) {
                    _saveStatus({
                        status: SYNC_NO_BEACON,
                        root: _root,
                        timeout: 60 * 60 * 1000
                    });
                    lastStatus = SYNC_NO_BEACON;
                }

                // Убираем остатки от похода за маячком
                if (frames._iframeBeacon) {
                    frames._iframeBeacon.parentNode.removeChild(frames._iframeBeacon);
                    frames._iframeBeacon = null;
                }

                window.clearTimeout(_timeoutBeacon);
                _beaconTimeoutTime = 50;

                // Маячок есть и с ним можно работать
                if (_beaconRoot <= beaconSelf) {

                    if (lastStatus === SYNC_NO_BEACON) {
                        _listener({
                            uuid: _uuid,
                            status: 'blocked'
                        });
                    } else if (lastStatus === SYNC) {
                        _listener({
                            uuid: _uuid,
                            status: 'up_to_date'
                        });
                    } else {
                        _listener({
                            uuid: _uuid,
                            status: lastStatus
                        });
                    }

                    _beaconRoot = void 0;

                    return;
                }

            }

            _setCookie({
                name: _guardCookieName,
                value: _uuid,
                expires: expires.toUTCString()
            });

            _sendLog({
                status: 'ok',
                action: 'start',
                result: result
            });

            _timeout = window.setTimeout(processTimeout, maxTime);

            setupIframe({
                _iframe: '_iframe',
                _source: _source
            });
        }

        function setupIframe(opts) {
            var iframe = document.createElement('iframe');
            iframe.src = opts._source;
            iframe.style.position = 'absolute';
            iframe.style.left = '-1000px';
            iframe.style.top = '-1000px';

            frames[opts._iframe] = iframe;

            document.body.appendChild(iframe);
        }

        function processTimeout() {
            _sendLog({
                status: 'error',
                result: 'failed',
                action: 'timeout'
            });

            _saveStatus({
                status: 'failed',
                root: _root,
                timeout: 20 * 60 * 1000
            });

            _cleanUp();

            _listener({
                uuid: _uuid,
                status: 'failed'
            });
        }

        function onMessage(event) {
            if (!~[_originSsoHost, _sourceHost].indexOf(event.origin)) {
                return;
            }

            if (event.data) {

                if (event.data.type === 'last_update') {
                    _beaconRoot = Number(event.data.last_update);
                    return;
                }

                processMessageData(event.data);
            }
        }

        function processMessageData(data) {
            var status = data.status;
            var timeout = data.timeout;
            var root = data.root;
            var message = data.message;

            _sendLog({
                status: 'ok',
                action: 'finish',
                result: status,
                message: message || null
            });

            _saveStatus({
                status: status,
                root: root,
                timeout: timeout
            });

            _cleanUp();

            _listener({
                uuid: _uuid,
                status: status
            });
        }

        function _cleanUp() {
            try {
                if (frames._iframe) {
                    frames._iframe.parentNode.removeChild(frames._iframe);
                    frames._iframe = null;
                }

                if (_timeout) {
                    window.clearTimeout(_timeout);
                }

                _clearCookie(_guardCookieName);
            } catch (error) {
                _sendLog({
                    status: 'error',
                    action: 'clean_up',
                    message: error
                });
            }
        }

        function _parseStatus(root, ssoStatus) {
            var roots = ssoStatus.split('#');
            var i = 0;
            var curr;

            while (i < roots.length) {
                curr = roots[i].split(':');

                if (curr[0] === root) {
                    return curr[1];
                }

                i++;
            }

            return null;
        };

        function getStatus() {
            var ssoStatusValue = _getCookie(_statusCookieName);
            var status = _parseStatus(_root, ssoStatusValue);

            return status;
        };

        function _getCookie(name) {
            var cookieMatcher = new RegExp('(?:(?:^|.*;\\s*)' + name + '\\s*\=\\s*([^;]*).*$)|^.*$');
            var cookieValue = window.document.cookie.replace(cookieMatcher, "$1");

            return cookieValue;
        }

        function _setCookie(opts) {
            var domain = _sso.split('.').slice(-2).join('.');

            document.cookie = opts.name + '=' + opts.value + ';domain=.'+ domain + ';path=/;Expires=' + opts.expires;
        }

        function _clearCookie(name) {
            _setCookie({
                name: name,
                value: '',
                expires: 'Thu, 01 Jan 1970 00:00:00 GMT'
            });
        }

        function _processStatus(ssoStatus) {
            var roots = ssoStatus.split('#');
            var result = {};
            var i = 0;
            var curr;

            if (!ssoStatus) {
                return result;
            }

            while (i < roots.length) {
                curr = roots[i].split(':');
                result[curr[0]] = curr[1];
                i++;
            }

            return result;
        };

        function _serializeStatus(statusObj) {
            var result = [];

            for (var domain in statusObj) {
                if (statusObj.hasOwnProperty(domain)) {
                    result.push(domain + ':' + statusObj[domain]);
                }
            }

            return result.join('#');
        };

        function _saveStatus(opts) {
            var status = opts.status;
            var root = opts.root;
            var expires = new Date((new Date()).getTime() + parseInt(opts.timeout, 10));
            var ssoStatusValue = _getCookie(_statusCookieName);
            var currentStatus = _processStatus(ssoStatusValue);

            currentStatus[root] = status;
            _setCookie({
                name: _statusCookieName,
                value: _serializeStatus(currentStatus),
                expires: expires.toUTCString()
            });
        }

        function _sendLog(opts) {
            var i = new Image();
            var fields = ['uuid=' + encodeURIComponent(_uuid)];

            fields.push('event=pull');
            fields.push('action=' + encodeURIComponent(opts.action));
            fields.push('status=' + encodeURIComponent(opts.status));
            fields.push('target=' + encodeURIComponent(window.location.hostname));
            fields.push('origin=' + encodeURIComponent(_root));

            if (opts.result) {
                fields.push('result=' + encodeURIComponent(opts.result));
            }

            if (opts.message) {
                fields.push('message=' + encodeURIComponent(opts.message));
            }

            i.src = 'https://' + _root + '/log?' + fields.join('&');
            i = null;
        }
    })());
})(window, window.document);
