(function (window, document) {
    var MDA2 = window.MDA2 || (window.MDA2 = (function () {
        var _guardCookieName = 'sso_uuid';
        var _statusCookieName = 'sso_status';
        var _iframe = null;
        var _timeout = null;
        var _uuid;
        var _root;
        var _sso;
        var _listener;
        var _originSsoHost;
        var _sourceHost;
        var _source;

        return {
            init: init,
            sync: sync
        };

        function init(opts) {
            _uuid = opts.uuid;
            _root = opts.root;
            _sso = opts.sso;
            _listener = opts.listener;
            _originSsoHost = 'https://' + _sso;
            _sourceHost = 'https://' + _root;
            _source = _sourceHost + '/pull?origin=' + encodeURIComponent(window.location.href) + '&uuid=' + encodeURIComponent(_uuid);

            if (window.addEventListener) {
                window.addEventListener('message', onMessage);
            } else {
                window.attachEvent('onmessage', onMessage);
            }
        };

        function sync() {
            var lastStatus = getStatus();
            var processUuid = _getCookie(_guardCookieName);
            var maxTime = 25000;
            var iframe;
            var expires = new Date((new Date()).getTime() + maxTime);

            if (processUuid) {
                _listener({
                    uuid: processUuid,
                    status: 'inprogress'
                });

                return;
            }

            if (lastStatus) {
                _listener({
                    uuid: _uuid,
                    status: lastStatus
                });

                return;
            }

            iframe = _iframe = document.createElement('iframe');
            iframe.src = _source;
            iframe.style.position = 'absolute';
            iframe.style.left = '-1000px';
            iframe.style.top = '-1000px';

            _setCookie({
                name: _guardCookieName,
                value: _uuid,
                expires: expires.toUTCString()
            });

            _sendLog({
                status: 'ok',
                action: 'start'
            });

            _timeout = window.setTimeout(processTimeout, maxTime);

            document.body.appendChild(iframe);
        }

        function processTimeout() {
            _sendLog({
                status: 'error',
                result: 'failed',
                action: 'timeout'
            });

            _saveStatus({
                status: 'failed',
                root: _root,
                timeout: 20 * 60 * 1000
            });

            _cleanUp();

            _listener({
                uuid: _uuid,
                status: 'failed'
            });
        }

        function onMessage(event) {
            if (!~[_originSsoHost, _sourceHost].indexOf(event.origin)) {
                return;
            }

            if (event.data) {
                processMessageData(event.data);
            }
        }

        function processMessageData(data) {
            var status = data.status;
            var timeout = data.timeout;
            var root = data.root;
            var message = data.message;

            _sendLog({
                status: 'ok',
                action: 'finish',
                result: status,
                message: message || null
            });

            _saveStatus({
                status: status,
                root: root,
                timeout: timeout
            });

            _cleanUp();

            _listener({
                uuid: _uuid,
                status: status
            });
        }

        function _cleanUp() {
            try {
                if (_iframe) {
                    _iframe.parentNode.removeChild(_iframe);
                }

                if (_timeout) {
                    window.clearTimeout(_timeout);
                }

                _clearCookie(_guardCookieName);
            } catch (error) {
                _sendLog({
                    status: 'error',
                    action: 'clean_up',
                    message: error
                });
            }
        }

        function _parseStatus(root, ssoStatus) {
            var roots = ssoStatus.split('#');
            var i = 0;
            var curr;

            while (i < roots.length) {
                curr = roots[i].split(':');

                if (curr[0] === root) {
                    return curr[1];
                }

                i++;
            }

            return null;
        };

        function getStatus() {
            var ssoStatusValue = _getCookie(_statusCookieName);
            var status = _parseStatus(_root, ssoStatusValue);

            return status;
        };

        function _getCookie(name) {
            var cookieMatcher = new RegExp('(?:(?:^|.*;\\s*)' + name + '\\s*\=\\s*([^;]*).*$)|^.*$');
            var cookieValue = window.document.cookie.replace(cookieMatcher, "$1");

            return cookieValue;
        }

        function _setCookie(opts) {
            var domain = _sso.split('.').slice(-2).join('.');

            document.cookie = opts.name + '=' + opts.value + ';domain=.'+ domain + ';path=/;Expires=' + opts.expires;
        }

        function _clearCookie(name) {
            _setCookie({
                name: name,
                value: '',
                expires: 'Thu, 01 Jan 1970 00:00:00 GMT'
            });
        }

        function _processStatus(ssoStatus) {
            var roots = ssoStatus.split('#');
            var result = {};
            var i = 0;
            var curr;

            if (!ssoStatus) {
                return result;
            }

            while (i < roots.length) {
                curr = roots[i].split(':');
                result[curr[0]] = curr[1];
                i++;
            }

            return result;
        };

        function _serializeStatus(statusObj) {
            var result = [];

            for (var domain in statusObj) {
                if (statusObj.hasOwnProperty(domain)) {
                    result.push(domain + ':' + statusObj[domain]);
                }
            }

            return result.join('#');
        };

        function _saveStatus(opts) {
            var status = opts.status;
            var root = opts.root;
            var expires = new Date((new Date()).getTime() + parseInt(opts.timeout, 10));
            var ssoStatusValue = _getCookie(_statusCookieName);
            var currentStatus = _processStatus(ssoStatusValue);

            currentStatus[root] = status;
            _setCookie({
                name: _statusCookieName,
                value: _serializeStatus(currentStatus),
                expires: expires.toUTCString()
            });
        }

        function _sendLog(opts) {
            var i = new Image();
            var fields = ['uuid=' + encodeURIComponent(_uuid)];

            fields.push('event=pull');
            fields.push('action=' + encodeURIComponent(opts.action));
            fields.push('status=' + encodeURIComponent(opts.status));
            fields.push('target=' + encodeURIComponent(window.location.hostname));
            fields.push('origin=' + encodeURIComponent(_root));

            if (opts.result) {
                fields.push('result=' + encodeURIComponent(opts.result));
            }

            if (opts.message) {
                fields.push('message=' + encodeURIComponent(opts.message));
            }

            i.src = 'https://' + _root + '/log?' + fields.join('&');
            i = null;
        }
    })());
})(window, window.document);
