const path = require('path');
const os = require('os');
const chai = require('chai');

const {getTusEnvironment, getPassportFrontendHost} = require('./test/hermione/testEnvironmentConfig');
const {errorPatterns} = require('@yandex-int/error-patterns');
const addCommands = require('./test/hermione/utils/addCommands');

const {getBrowserConfigs} = require('./test/hermione/browsers');

const loadPageObject = require('./test/hermione/page-objects/page-object').loadPageObject;
const PO = {
    desktop: loadPageObject('desktop')
    // 'touch-phone': loadPageObject('touch-phone')
};

const HTML_REPORT_DIR = 'hermione-report';

global.assert = chai.assert;

// переменная нужна для работы плагина hermione-debug
process.env.IPBUS_BROKER_RUNDIR =
    process.env.IPBUS_BROKER_RUNDIR || path.join(os.tmpdir(), `hermione-ipbus-${process.pid}`);

process.env.TEST_ENV = process.env.TEST_ENV || 'test';
process.env.TEST_TLD = process.env.TEST_TLD || 'ru';

module.exports = {
    baseUrl: getPassportFrontendHost(),

    gridUrl: 'http://passport-frontend@sw.yandex-team.ru:80/v0',
    // gridUrl: 'http://selenium:selenium@sg.yandex-team.ru:4444/wd/hub',
    browsers: getBrowserConfigs(),
    sets: {
        desktop: {
            files: [
                'test/hermione/features/**/*(?<!touch).hermione.js',
                'test/hermione/features/**/*@common.hermione.js',
                'test/hermione/features/**/*@desktop.hermione.js'
            ],
            ignoreFiles: ['**/{screens,test-data}/**'],
            browsers: [
                'chrome-desktop',
                'chrome-desktop-1920',
                // 'edge-desktop',
                'ipad',
                'ipad-portrait'
                // 'firefox'
            ]
        },
        'touch-phone': {
            files: [
                'test/hermione/features/**/*(?<!desktop).hermione.js',
                'test/hermione/features/**/*@common.hermione.js',
                'test/hermione/features/**/*@touch.hermione.js'
            ],
            ignoreFiles: ['**/{screens,test-data}/**'],
            browsers: [
                'iphone',
                'iphoneX',
                'yandex-browser-phone',
                'chrome-phone'
                // 'safari13'
            ]
        }
    },

    sessionsPerBrowser: 1,
    testsPerSession: 1,

    retry: 5,
    saveHistory: true,

    httpTimeout: 30000,
    urlHttpTimeout: 30000,
    // pageLoadTimeout: 20000,
    sessionRequestTimeout: 200000,
    sessionQuitTimeout: 5000,
    testTimeout: 240000,
    waitTimeout: 50000,
    screenshotOnRejectTimeout: 5000,
    screenshotsDir: (test) => path.join(path.dirname(test.file), 'screens', test.id(), test.browserId),

    system: {
        patternsOnReject: [
            /timeout/i,
            /timedout/i,
            /timed out/i,
            /It may have died/,
            /CLIENT_STOPPED_SESSION/,
            /was terminated due to CLIENT_GONE/,
            /cannot forward the request/,
            /FORWARDING_TO_NODE_FAILED/,
            /PROXY_REREGISTRATION/,
            /not available and is not among the last/,
            /The specified window could not be found/,
            /ESOCKETTIMEDOUT/,
            /route not found/
        ],
        parallelLimit: 15
    },

    plugins: {
        'hermione-global-hook': {
            beforeEach: async function() {
                await this.browser.deleteCookie();
                // TODO: const meta = await this.browser.getMeta();
                const meta = {platform: 'desktop'};

                this.PO = PO[meta.platform];
                this.currentPlatform = meta.platform;
                // TODO: return this.browser.yaLocalStorageClear();
            }
        },
        'html-reporter/hermione': {
            enabled: true,
            path: HTML_REPORT_DIR,
            defaultView: 'failed',
            lazyLoadOffset: 2500,
            errorPatterns,
            metaInfoBaseUrls: {
                file: '',
                sessionId: 'https://sw.yandex-team.ru/log/'
            },
            saveErrorDetails: true,
            customGui: {
                'switch-passport-environment': [
                    {
                        type: 'radiobutton',
                        controls: [
                            {
                                label: 'Dev',
                                value: 'dev'
                            },
                            {
                                label: 'Test',
                                value: 'test'
                            },
                            {
                                label: 'RC',
                                value: 'rc'
                            },
                            {
                                label: 'Prod',
                                value: 'prod'
                            }
                        ],
                        initialize: async ({ctx}) => {
                            ctx.controls.forEach((control) => {
                                control.active = process.env.TEST_ENV === control.value;
                            });
                        },
                        action: async ({hermione, control}) => {
                            process.env.TEST_ENV = control.value;
                            const {config} = hermione;
                            // TODO: Научиться прокидывать getTusEnvironment в плагин для TUS в tus_env
                            config.getBrowserIds().forEach((browserId) => {
                                config.forBrowser(browserId).baseUrl = getPassportFrontendHost();
                            });
                        }
                    }
                ],
                'switch-passport-tld': [
                    {
                        type: 'radiobutton',
                        controls: [
                            {
                                label: '.RU',
                                value: 'ru'
                            },
                            {
                                label: '.COM.TR',
                                value: 'com.tr'
                            }
                        ],
                        initialize: async ({ctx}) => {
                            ctx.controls.forEach((control) => {
                                control.active = process.env.TEST_TLD === control.value;
                            });
                        },
                        action: async ({hermione, control}) => {
                            process.env.TEST_TLD = control.value;
                            const {config} = hermione;
                            config.getBrowserIds().forEach((browserId) => {
                                config.forBrowser(browserId).baseUrl = getPassportFrontendHost();
                            });
                        }
                    }
                ]
            }
        },
        '@yandex-int/hermione-auth-commands': {
            enabled: true,

            // tus_consumer. Обязательное поле.
            // Значение по умолчанию не определено.
            // Следует заводить своих консьюмеров для различных проектов.
            // Про флоу заведения и использования консьюмеров - [ниже](#tus_consumer).
            tus_consumer: 'kopusha',

            // Окружение Паспорта. Соответствие между значениями данной опции и окружениями
            //   следует смотреть здесь - https://wiki.yandex-team.ru/test-user-service/#env
            // По умолчанию "test", что соответствует окружению https://passport-test.yandex.ru
            env: getTusEnvironment(),

            // При создании новых тестовых аккаунтов TUS ожидает, что логины будут начинаться
            //   с "yandex-team-" или "yndx-".
            // Данная необязательная опция предназначена для избегания засилия таких префиксов
            //   при вызовах команд авторизации.
            // В случае неустановки данной опции соответствующие префиксы нужно будет указывать
            //   при вызове каждой команды авторизации.
            // В случае, если некоторый логин уже начинается с "yandex-team-" или "yndx-"
            //   данная опция игнорируется.
            // По умолчанию - пустая строка. Может быть переопределено при вызовых отдельных команд.
            //   через опцию `prefix`.
            loginPrefix: 'yndx',

            // Значение по умолчанию для количества аккаунтов в группе.
            // Используется при вызовах команды `authAny`.
            // Один из аккаунтов указаной группы будет заблокирован от использования в других тестах.
            // Значение следует выбирать соответствующим параллельности запускаемых тестов.
            // По умолчанию равно 1. Может быть перерпределено в `authAny` через опцию `size`.
            groupSize: 1,

            // Значение по умолчанию для времени блокировки аккаунта из группы в секундах.
            // Используется при вызовах команды `authAny`.
            // Один из аккаунтов указаной группы будет заблокирован от использования в других
            //   тестах на указанное время.
            // Значение следует выбирать соответствующим таймаутам выполнения запускаемых тестов.
            // По умолчанию равно 0, что значит не блокировать никогда.
            // Может быть кастомизировано при вызовах `authAny` через опцию `lockFor`.
            groupLockFor: 0,

            // Логировать логин и пароль использованного для авторизации тестового аккаунта.
            // Может быть полезно для получения пароля от тестового аккаунта для
            // целей дополнительной ручной настройки аккаунта.
            //
            // Альтернативным вариантом является использование команд show или list из tus-cli.
            //
            // Также данная опция конфигурации может быть установлена через переменную окружения
            // `hermione_auth_commands_log_auth=true` или опцию `--auth-commands-log-auth=true`,
            // для включения логирования в единичных запусках. Также одноименную опцию
            // принимают команды authAny и auth.
            //
            // Не рекомендуется использовать эту опцию вне локальной разработки.
            // При этом, при использовании из другого плагина hermione-auth-on-record-commands
            // включение данной опции позволяет логировать только при снятии или
            // переснятии дампов.
            logAuth: false,

            // Различные опции, совпадающие с опциями @yandex-int/si.ci.requests
            // будут применятся для всех запросов к TUS.
            requestOptions: {}
        },
        '@yandex-int/hermione-debug': {
            novncUrl: 'https://sw.yandex-team.ru/vnc',
            vncHost: 'sw.yandex-team.ru',
            vncPathPrefix: 'v0/vnc/',
            enabled: false, // Не нужно запускать debug по умолчанию

            noTimeouts: true, // Выключить таймауты сессии и времени работы теста
            retryOnSuccess: false, // Перезапускать тест, если он завершился успешно
            vncOnFail: false, // Открывать VNC-клиент при падении теста
            vncBeforeTest: false, // Открывать VNC-клиент перед стартом теста
            pauseOnVnc: false, // Останавливать прогон теста после открытия VNC-клиента (enter для продолжения)
            recordVideo: false, // Записать видео прогона теста (ссылка будет выведена в лог)
            // assignedCapabilities: { // Эти capabilities будут подмешаны к desiredCapabilities браузера
            //     'chrome-phone': {
            //         version: 'phone-67.0-vnc',
            //     }
            // },
            localVncProxy: {
                // Список браузеров, для которых VNC работает через проксирование VNC-трафика с удалённого сервера на локальный VNC-сервер
                safari13: {
                    // Список опций, которые будут подмешаны в опции браузера, если включена или опция 'vncBeforeTest', или опция 'vncOnFail'
                    gridUrl: 'http://ios:ios@sg.yandex-team.ru:4444/wd/hub'
                }
            },
            browsers: /.*/ // Браузеры в которые необходимо добавить капабилити для дебага
        },
        'hermione-safari-commands': {
            enabled: true,
            browsers: {
                safari13: {
                    commands: [
                        'url',
                        'click', // TODO: не работает
                        'screenshot',
                        'orientation',
                        'swipe',
                        'touch',
                        'dragAndDrop',
                        'deviceClickBack'
                    ]
                }
            }
        },
        // Важно! Этот плагин должен подключаться самым последним, иначе он работать не будет
        'hermione-wdio-migrator': {
            // плагин позволяет использовать новую версию гермионы вместе со старыми плагинами
            enabled: true
        }
    },
    prepareBrowser: function(browser) {
        browser.addCommand('getConfig', () => this);
        addCommands(browser, '../commands');
    }
};
