var config = require('./configs/current');
var configCommon = require('./configs/common');
var inherit = require('inherit');
var when = require('when');
var fs = require('fs');
var uuid = require('uuid');
var morgan = require('morgan');
var cookieParser = require('cookie-parser');
var bodyParser = require('body-parser');
var multiparty = require('connect-multiparty');
var contentType = require('content-type');
var helmet = require('helmet');
const noCache = require('nocache');

var requestsLogFile = fs.createWriteStream(config.paths.logs.requests, {flags: 'a', encoding: 'utf8'});
const serialize = require('serialize-javascript');
const lookup = require('./lib/getLookup.js');
const trackRe = /^\w{16,34}$/;
const pathRe = /^\/*/;
const getDefaultUser = require('./routes/common/checkAuth/getDefaultUser');
const patchRequestWithDebugInfo = require('./lib/patchRequestWithDebugInfo.js');
const isLocalDev = config.version === 'local';

// eslint-disable-next-line no-useless-escape, max-len
const WEBVISOR_AND_METRICKA_DOMAINS = /^([^\/]+\.)?(passport.yandex\.(?:ru|ua|by|kz|com|com\.tr)|webvisor\.com|metri[ck]a\.yandex\.(com|ru|by|com\.tr))$/;

let serviceTicketsSetup = require('./routes/common/serviceTicketsSetup');

process.on('unhandledRejection', (reason, promise) => {
    console.error('Unhandled Rejection at:', promise, 'reason:', reason);
});

// throw new Error('Oh Hi, Mark!');

requestsLogFile.on('error', function() {
    // throw new Error(err);
});

if (process.env.IP_VERSION === 'ipv6only') {
    require('dns-graceful-stack-switch')(6);
}

// eslint-disable-next-line no-unused-vars
var dnscache = require('dnscache')({
    enable: true,
    ttl: 300,
    cachesize: 1000
});

var PView = require('pview');
var Controller = require('./lib/controller');
var RenderingLayoutView = require('./blocks/layout/RenderingLayoutView');
var express = require('express');
var app = (module.exports = express());
const initDefaults = require('./lib/controller/initDefaults');

app.use(cookieParser());

if (isLocalDev) {
    app.use((req, res, next) => {
        req.headers['x-real-ip'] = '127.0.0.1';
        req.headers['x-real-scheme'] = 'https';

        next();
    });
}

app.use(function getIsYandex(req, res, next) {
    const iP = req.get('x-real-ip');

    let isYandexInternalNetwork = false;

    if (iP) {
        const net = lookup.getIpTraits(iP);

        res.locals.isYandexInternalNetwork = isYandexInternalNetwork = net && Boolean(net.yandex_net);
        res.locals.isYandex = isYandexInternalNetwork && Boolean(net.yandex_staff);
    }

    res.locals.nodeVersion = process.version;

    if (isYandexInternalNetwork) {
        patchRequestWithDebugInfo(req);
    }

    return next();
});

var Logger = require('plog');
var logSettings = Logger.configure();

logSettings
    .minLevel(config.loglevel.toLowerCase())
    .setFormatter(logSettings.getFormatterByName('production'))
    .setHandler(logSettings.getHandlerByName('buffering'));

if ('development' === app.get('env') || isLocalDev) {
    logSettings.setFormatter(logSettings.getFormatterByName('dev')).setHandler(logSettings.getHandlerByName('console'));
}

if (process.env.RUNNER === 'tests') {
    logSettings
        .minLevel('critical')
        .setFormatter(logSettings.getFormatterByName('dev'))
        .setHandler(logSettings.getHandlerByName('console'));

    serviceTicketsSetup = (req, res, next) => next();
}

PView.setRenderer(PView.yateRenderer) // Настройка рендеринга pview
    .setTemplatesDir(`${__dirname}/templates`);

require('putils').csrf.setSalt('Ophah3ean3feaw2uv5rinoo5udee1oongei2phieZ3ohpie3bier3Eth8ohghoos');
require('putils')
    .i18n.setAllowedLangs(config.langs)
    .setKeysets([
        'Common',
        'EULA',
        'Frontend',
        'SimpleReg',
        'Enabling2fa',
        'Semiauto',
        'Social',
        'ProfileAccess',
        'Time',
        'Restoration',
        'Restore',
        'ProfileSocial',
        'Phones',
        'ServicesSubscriptions',
        'Navigation'
    ])
    .loadFromDir(`${__dirname}/loc`);
config.langs.forEach((lang) => require(`dayjs/locale/${lang}`));

var cluster = require('cluster');
var tools = require('prequest/tools');

// Init yate externals
require('./blocks/_externals/node-externals.js')(PView.getYateRenderer());

morgan.token('worker', function() {
    // TODO: remove me (PASSP-9509)
    var worker = '';

    if (cluster.isWorker) {
        worker = `${cluster.worker.id}:${cluster.worker.process.pid}`;
    }

    return worker;
});

morgan.token('pathname', function(req) {
    const parsed = new URL(req.originalUrl.replace(pathRe, ''), 'https://abc.def/');
    const {pathname, searchParams} = parsed;
    const mode = searchParams.get('mode');
    const action = searchParams.get('action') || (req.body && req.body.action) || null;

    if (mode === 'embeddedauth') {
        if (action !== null) {
            return `${pathname}_mode__${mode}_action__${action}`;
        }
    }

    if (pathname === '/passport') {
        if (mode !== null) {
            return `${pathname}_mode__${mode}`;
        }
    }

    return pathname;
});

morgan.token('body', function(req) {
    // TODO: remove me (PASSP-9509)
    return JSON.stringify(tools.clearLog(req.body));
});

morgan.token('localdate', tools.localdate); // TODO: remove me (PASSP-9509);

morgan.token('track_id', function(req, res) {
    // TODO: remove me (PASSP-9509)
    return (
        (res.locals.track_id && res.locals.track_id.id) ||
        (req.body && req.body.track_id) ||
        (req.nquery && req.nquery.track_id) ||
        (req.query && req.query.track_id) ||
        'notrack'
    );
});

morgan.token('logID', function(req) {
    // TODO: remove me (PASSP-9509)
    return req.logID;
});

morgan.token('err', function(req, res) {
    // TODO: remove me (PASSP-9509)
    var err = '';

    if (res.err) {
        err = res.err.stack ? JSON.stringify(res.err.stack) : JSON.stringify(res.err);
    }

    return err;
});

var LegacyView = inherit(PView, {
    __constructor(name, options) {
        this.name = `legacy.view.${name}`;
        this._options = options;

        this.__base.apply(this, arguments);
    },
    _compile() {
        return this._options;
    }
});
var LegacyLayoutView = inherit(RenderingLayoutView, {
    __constructor(controller, name) {
        this.name = name;
        this.template = `${name}.%lang%.js`;

        this.__base(controller);
    }
});

app.engine('js', function(path, options, callback) {
    // TODO: нужно получать имя модуля по нормальному
    var filename = path
        .split('/')
        .pop()
        .split('.');

    filename.pop(); // Drop extension
    filename.pop(); // Drop locale
    var name = filename.join('.');

    // _reqres is passed via app.use call
    var request = options._reqres.req;
    var response = options._reqres.res;

    delete options._reqres;
    delete options._locals._reqres;

    var logID = request.logID;
    var controller = request._controller;
    var auth = controller.getAuth();
    var display = request.body.display || request.query.display;

    if (display === 'touch') {
        options.display = display;
    }

    var layout = new LegacyLayoutView(controller, name);

    layout.append(new LegacyView(name, options));

    auth.sessionID({
        multisession: 'yes',
        full_info: 'yes',
        get_family_info: 'yes',
        get_public_id: 'yes',
        get_public_name: 'yes',
        allow_child: 'yes',
        emails: 'getdefault',
        // @see https://wiki.yandex-team.ru/passport/dbmoving/#tipyatributov
        attributes: '27,28,32,34,1003,1005,1011',
        aliases: '1,5,6,13,21'
    })
        .then(
            function(sessionInfo) {
                return layout.render(sessionInfo);
            },
            function(err) {
                const reqOriginalUrl = request.originalUrl;

                if (err.code !== 'need_resign') {
                    return when.reject(err);
                }

                if (reqOriginalUrl.indexOf('/auth/social') === 0) {
                    return layout.render();
                }

                if (reqOriginalUrl.indexOf('/auth/update') === 0) {
                    return layout.render();
                }

                if (request.query && request.query.mode === 'error') {
                    return layout.render();
                }

                const mdaCookie = controller.getCookie('mda');

                if (mdaCookie === '1') {
                    return auth.resign();
                }

                if (app.get('env') === 'production') {
                    return auth.resign();
                }

                return controller.getUatraits().then(function(uatraits) {
                    if (uatraits.isMobile) {
                        return layout.render();
                    } else {
                        return auth.resign();
                    }
                });
            }
        )
        .then(
            function(rendered) {
                callback(null, rendered);
            },
            function(err) {
                Logger.error()
                    .logId(logID) // Passed via locals
                    .type('passport', 'app', 'renderer')
                    .write('Template error on %s:', name, err);

                callback(err);
            }
        )
        .then(function() {
            const statData = controller.getLocalsField('statData');

            if (!statData) {
                // console.error('No statbox on', name);
                return;
            }

            const req = request;
            const res = response;
            const log = Object.assign(
                {},
                {
                    action: 'opened',
                    mode: name,
                    url: req.url,
                    user_agent: req.headers['user-agent'],
                    yandexuid: res._yandexuid || req.cookies.yandexuid,
                    ip: req.headers['x-real-ip'],
                    from: (req.nquery && req.nquery.from) || null,
                    origin: (req.nquery && req.nquery.origin) || null,
                    referer: req.headers.referer || null,
                    retpath: (req.nquery && req.nquery.retpath) || null,
                    host: req.hostname
                },
                statData
            );

            controller.writeStatbox(log);
        });
});

app.engine('jsx', (path, options, callback) => {
    const filename = path
        .split('/')
        .pop()
        .split('.');

    filename.pop(); // Drop extension
    filename.pop(); // Drop locale

    const name = filename.join('.');
    const template = require(path);
    const {req, res} = options._reqres;
    const {logID} = req;

    // get csrf
    req._controller
        .getCsrfToken()
        .then((csrf) => {
            if (!res.locals.store.common) {
                res.locals.store.common = {};
            }

            if (!res.locals.store.common.csrf) {
                res.locals.store.common.csrf = csrf;
            }

            if (!res.locals.store.common.track_id) {
                res.locals.store.common.track_id = res.locals.track_id;
            }

            res.locals.serializedStore = serialize(res.locals.store, {isJSON: true});
            res.locals.page = name;

            try {
                template.renderPage(res);
            } catch (error) {
                throw error;
            }
        })
        .catch((error) => {
            Logger.error()
                .logId(logID) // Passed via locals
                .type('passport', 'app', 'renderer')
                .write('Template error on %s:', path, error);

            callback(error);
        });
});

if (app.get('env') === 'development' || isLocalDev) {
    app.use(
        '/st',
        (req, res, next) => {
            var origin = req.headers.origin;

            if (origin) {
                res.setHeader('Access-Control-Allow-Origin', origin);
            }
            res.setHeader('Access-Control-Allow-Methods', 'GET');
            res.setHeader('Access-Control-Allow-Credentials', 'true');

            return next();
        },
        express.static(`${__dirname}/public`)
    );
    app.use('/monitoring', function(req, res) {
        return res.status(204).end();
    });
}

app.use(function setupLogID(req, res, next) {
    var logID = req.header('x-request-id') || '0000000';

    req.logID = res.locals.logId = logID.slice(-12);
    return next();
});

app.use(function getDetectors(req, res, next) {
    req.uatraitsDetector = Controller.expressUatraitsDetector;
    req.langDetector = Controller.expressLangDetector;
    // Pass request and response to app.engine to instantiate a controller
    res.locals._reqres = {
        req,
        res
    };

    return next();
});

app.use(function getRegionId(req, res, next) {
    const iP = req.get('x-real-ip');
    const xYproxyHeaderIp = req.get('x-yproxy-header-ip');
    const xForwardedForY = req.get('X-Forwarded-For-Y');

    if (iP) {
        try {
            res.locals.regionId = lookup.getRegionIdByIp(iP);
            res.locals.countryId = lookup.getCountryId(res.locals.regionId);
        } catch (err) {
            Logger.error()
                .logId(req.logID)
                .type('passport', 'app', 'getRegionId')
                .write(err);
        }
    }

    if (xYproxyHeaderIp) {
        Logger.info()
            .logId(req.logID)
            .type('passport', 'app', 'xYproxyHeaderIp')
            .write(xYproxyHeaderIp);
    }

    if (xForwardedForY) {
        Logger.info()
            .logId(req.logID)
            .type('passport', 'app', 'xForwardedForY')
            .write(xForwardedForY);
    }

    return next();
});

// Middlewares to improve security
app.use(function(req, res, next) {
    res.locals.nonce = uuid.v4();
    next();
});
app.use(noCache());
app.use(helmet.contentSecurityPolicy(config.helmet.csp));
app.use((req, res, next) => {
    if (res.locals.isYandex && WEBVISOR_AND_METRICKA_DOMAINS.test(req.hostname)) {
        return next();
    }
    return helmet.frameguard(config.helmet.frameguard)(req, res, next);
});
app.use(helmet.hsts(config.helmet.hsts));
app.use(helmet.hidePoweredBy());
app.use(helmet.ieNoOpen());
app.use(helmet.noSniff());
app.use(helmet.dnsPrefetchControl());
app.use(helmet.xssFilter());
app.use(
    helmet.referrerPolicy({
        policy: 'origin'
    })
);

// noinspection JSValidateTypes
app.use(function(req, res, next) {
    res.setHeader('P3P', 'policyref="/w3c/p3p.xml", CP="NON DSP ADM DEV PSD IVDo OUR IND STP PHY PRE NAV UNI"');
    return next();
});

app.use(function(req, res, next) {
    if (req.method === 'GET' && req.headers['x-real-scheme'] === 'http') {
        res.redirect(302, `https://${req.hostname}${req.originalUrl}`);
    } else {
        next();
    }
});

app.use(
    morgan(':logID :method :pathname :status :response-time', {
        stream: requestsLogFile
    })
);

// Parameters on the application/x-www-form-urlencoded MIME type are ignored.
// In particular, this MIME type does not support the charset parameter.
// http://www.w3.org/TR/html5/forms.html#application/x-www-form-urlencoded-encoding-algorithm
app.use(function(req, res, next) {
    var defaultContentType;

    if (req.method !== 'POST') {
        return next();
    }

    defaultContentType = contentType.format({
        type: 'application/x-www-form-urlencoded'
    });

    try {
        var contentTypeObj = contentType.parse(req.headers['content-type']);

        if (contentTypeObj.type === 'application/x-www-form-urlencoded') {
            req.headers['content-type'] = defaultContentType;
        }

        return next();
    } catch (err) {
        // Поддерживаем старые бары и другое легаси, которое не присылает content-type
        req.headers['content-type'] = defaultContentType;
        return next();
    }
});

app.use('/restoration/semi_auto', multiparty());
app.use('/restoration/semi_auto/submit', multiparty());
app.use('/profile/avatars/legacy/', multiparty());
app.use(
    bodyParser.urlencoded({
        extended: false
    })
);
app.use(function requestBodyLogger(req, res, next) {
    // 'INFO request.body [:localdate] :logID :track_id :method :url :body'
    Logger.info()
        .logId(req.logID)
        .type('passport', 'app')
        .write('REQ: %s %s %j', req.method, req.url, tools.clearLog(req.body));
    next();
});

app.use(function(req, res, next) {
    if (req.method !== 'POST') {
        return next();
    }

    if (!(req.body && req.body.track_id)) {
        return next();
    }

    if (!trackRe.test(req.body.track_id)) {
        req._honeypot = true;
        delete req.body.track_id;

        Logger.info()
            .logId(req.logID)
            .type('passport', 'app', 'bad_track_id')
            .write(req.body.track_id);
    }

    return next();
});

app.use(function nquery(req, res, next) {
    if (!req.nquery) {
        // express.param пытается urlDecode'ить query и возвращает оригинальную строку,
        // если decodeUriComponent кидает эксцепшен
        // нодовский querystring, который используется в url.parse в этом случае пытается
        // декодить строку другими методами
        //
        // Правильнее всегда сначала смотреть в nquery, потом в req.query

        /* eslint-disable max-len */
        // @see node-querystring used by express {@link https://github.com/visionmedia/node-querystring/blob/639ccb6bf7e8da2c2d4c22dc8600cba5845fa25d/index.js#L360-L366}
        // @see querystring bundled with node {@link https://github.com/joyent/node/blob/8d3bc88bbe3c0690f433a97c1df33fca099f18be/lib/querystring.js#L188-L194}
        /* eslint-enable max-len */

        // req.nquery = url.parse(req.url, true).query;
        req.nquery = req.query;
    }
    next();
});

// X-Forwarded-For нет, 'trust proxy' бесполезно
app.disable('trust proxy');
app.use(function(req, res, next) {
    Object.defineProperty(req, 'ip', {value: req.get('x-real-ip')});
    next();
});

app.set('view engine', 'js');
app.set('views', `${__dirname}/templates`);
// settings.* в шаблонах
app.set('paths', config.paths);
app.set('pathsStr', JSON.stringify(config.paths));
app.set('env', app.get('env'));
app.set('intranet', process.env.INTRANET === 'intranet');
app.set('doNotShow2FA', config.doNotShow2FA);

app.get('/ping', function(req, res) {
    res.status(200).send('pong');
});

app.use(serviceTicketsSetup);
initDefaults(app);

app.all('*', function(req, res, next) {
    const controller = req._controller;
    const _yandexuid = controller.getCookie('yandexuid');

    Logger.info()
        .logId(req.logID)
        .type('passport', 'app', 'yandexuid')
        .write(_yandexuid);

    next();
});

app.use(function(req, res, next) {
    if (req.method !== 'GET') {
        return next();
    }

    const controller = req._controller;
    const auth = controller.getAuth();

    return auth
        .sessionID({
            multisession: 'yes',
            full_info: 'yes',
            get_family_info: 'yes',
            get_public_id: 'yes',
            get_public_name: 'yes',
            allow_child: 'yes',
            emails: 'getdefault',
            // @see https://wiki.yandex-team.ru/passport/dbmoving/#tipyatributov
            attributes: '27,28,32,34,1003,1005,1011',
            aliases: '1,5,6,13,21'
        })
        .then(function(response) {
            if (response) {
                req.blackbox = response;
            }

            return next();
        })
        .catch(function() {
            return next();
        });
});

app.use(function(req, res, next) {
    if (req.method !== 'GET' || !req.blackbox) {
        return next();
    }

    const defaultUid = req.blackbox.default_uid;
    const users = req.blackbox.users;
    const defaultUser = getDefaultUser(users, defaultUid);

    let isPortal = false;

    let isWSUser = false;

    let isPDDUser = false;

    let isSocial = false;

    let isNeoPhonish = false;

    let isLite = false;

    let isSuperLite = false;

    let hasPassword = false;

    let metricsUserType;

    if (defaultUser) {
        hasPassword = defaultUser.attributes && Boolean(defaultUser.attributes[1005]);
        isWSUser = defaultUser.attributes && Boolean(defaultUser.attributes[1011]);
        isPDDUser = defaultUser.uid && defaultUser.uid.hosted;
        isPortal = defaultUser.aliases && defaultUser.aliases.hasOwnProperty('1');
        isSocial = defaultUser.aliases && defaultUser.aliases.hasOwnProperty('6') && !isPortal;
        isNeoPhonish = defaultUser.aliases && defaultUser.aliases.hasOwnProperty('21') && !isPortal;
        isLite = defaultUser.aliases && defaultUser.aliases.hasOwnProperty('5') && !isPortal;
        isSuperLite = isLite && !hasPassword;
    }

    if (isPDDUser || isWSUser) {
        metricsUserType = 'pdd';
    } else if (isPortal) {
        metricsUserType = 'p';
    } else if (isSocial) {
        metricsUserType = 'soc';
    } else if (isNeoPhonish) {
        metricsUserType = 'np';
    } else if (isLite) {
        metricsUserType = 'l';
    } else if (isSuperLite) {
        metricsUserType = 'sl';
    }

    // Проверить привязан ли логин сотрудника на стаффе (сид 668 И 669).
    res.locals.isYandexoid = defaultUser && defaultUser.aliases && defaultUser.aliases.hasOwnProperty('13');
    res.locals.userType = {
        isWSUser,
        isPDDUser,
        isPortal,
        isSocial,
        isNeoPhonish,
        isLite,
        isSuperLite,
        metricsUserType
    };

    if (res.locals.isYandexoid || res.locals.isYandex) {
        res.locals.beatleUrl = configCommon.beatleUrl;
    }

    return next();
});

// routes
if (process.env.INTRANET === 'intranet') {
    require('./router.intranet.js')(app);
} else {
    require('./router.js')(app);
}

app.use(function getDetectors(err, req, res, next) {
    req.uatraitsDetector = Controller.expressUatraitsDetector;
    req.langDetector = Controller.expressLangDetector;
    // Pass request and response to app.engine to instantiate a controller
    res.locals._reqres = {
        req,
        res
    };

    return next(err);
});

// errors
require('./routes/common/global-errors.js')(app);

var server = app.listen(process.env.PORT || process.env.npm_package_config_port || 3000, '127.0.0.1', function() {
    if (!cluster.isWorker) {
        /* eslint-disable no-console */
        console.log('opened server on', server.address());
        console.log('version:', process.version);
        /* eslint-enable no-console */
    }
});
