var Field = require('pform/Field');
var _ = require('lodash');

/**
 * @extends Field
 */
module.exports = require('inherit')(
    Field,
    {
        __constructor: function() {
            this._init('captcha')
                .setName('answer')
                .setLabel('%field_answer')
                .addError(new Field.Error('missingvalue', '%answer_errors_missingvalue'))
                .addError(new Field.Error('captchalocate', '%answer_errors_captchalocate'))
                .addError(new Field.Error('incorrect', '%answer_errors_incorrect'));

            this._key = new this.__self.KeyField();
            this._mode = 'text';
        },

        /**
         * Fetch a captcha mode from POSTed data
         * @param {object} formData
         * @returns {string}
         * @private
         */
        _parseMode: function(formData) {
            return String(formData['captcha_mode']);
        },

        /**
         * Return current mode: text or audio
         *
         * @returns {string}
         */
        getMode: function() {
            return this._mode;
        },

        /**
         * Set current mode to either text or audio
         *
         * @returns {CaptchaField}
         */
        setMode: function(mode) {
            if (['text', 'audio'].indexOf(mode) > -1) {
                this._mode = mode;
            } else {
                this._mode = 'text';
            }

            return this;
        },

        /**
         * Set the key used to fetch a captcha
         * @param {string} key
         * @returns {CaptchaField}
         */
        setKey: function(key) {
            this._key.setValue(key);
            return this;
        },

        compile: function(lang, api) {
            // Compile calls captchaGenerate and not audioCaptchaGenerate no matter the mode,
            // because audio is fetched by js and there should be a captcha in case js fails

            var that = this;
            var compiled = this.__base.apply(this, arguments);

            return api.captchaGenerate(that.getOptions()).then(function(captcha) {
                that.setKey(captcha.body.key);
                return _.merge({}, compiled, _.pick(captcha.body, 'image_url', 'key'), {
                    voice_url: captcha.body.voice.url,
                    voice_intro_url: captcha.body.voice.intro_url,
                    mode: that.getMode(),
                    static: that.getStaticPath(),
                    countryFromAudioWhiteList: that.checkCountryAudioWhiteList()
                });
            });
        },

        checkCountryAudioWhiteList: function() {
            var whitelist = require('../../../configs/current').audioCaptcha.whitelist;
            var opts = this.getOptions() || {};
            var country = (opts.country && opts.country.toUpperCase()) || '';

            if (!country) {
                // если страну не передавали, то считаем, что все хорошо
                return true;
            }

            if (country && whitelist.indexOf(country) !== -1) {
                return true;
            } else {
                return false;
            }
        },

        getStaticPath: function() {
            return require('../../../configs/current').paths.static;
        },

        isEmpty: function(formData) {
            return !this._parseValue(formData);
        },

        onEmpty: function(formData) {
            this.setMode(this._parseMode(formData));

            if (this.isRequired()) {
                this.getErrorByCode('missingvalue').setActive();
            }
        },

        validate: function(formData, api) {
            var answer = this._parseValue(formData);

            if (!answer) {
                this.getErrorByCode('missingvalue').setActive();
                return false;
            }

            if (this._key.isEmpty(formData)) {
                return ['captchalocate'];
            }

            return api
                .captchaCheck({
                    answer: answer,
                    key: this._key._parseValue(formData)
                })
                .then(function(result) {
                    if (!result.body) {
                        throw new Error('{ field: "password", body: <response> } expected');
                    }

                    if (!('correct' in result.body)) {
                        throw new Error('Expected api response to contain "correct" field');
                    }

                    if (result.body.correct) {
                        return [];
                    }

                    return ['incorrect'];
                });
        },

        onInvalid: function(errors, formData) {
            this.setMode(this._parseMode(formData));
            this.setErrorsActive(errors);
        },

        /**
         * Captcha should not save state
         */
        onValid: function(formData) {
            this.setMode(this._parseMode(formData));
        }
    },
    {
        /**
         * @extends Field
         */
        KeyField: require('inherit')(Field, {
            __constructor: function() {
                this._init('key');
            },

            isEmpty: function(formData) {
                return !this._parseValue(formData);
            }
        })
    }
);
