/* eslint-env node */
var _ = require('lodash');
var when = require('when');
var inherit = require('inherit');

var Form = require('pform');
var Field = require('pform/Field');
var PhoneConfirmField = require('../phone-confirm/phone-confirm.field');
var QuestionField = require('../question/question.field');
var UserQuestionField = require('../user-question/user-question.field');
var AnswerField = require('../answer/answer.field');
var CaptchaField = require('../captcha/captcha.field');

/**
 * @extends Field
 */
var HumanConfirmationFieled = inherit(Field, {
    __constructor: function() {
        var self = this;

        this._init('human-confirmation')
            .addError(new Field.Error('unconfirmed', '%field_human-confirmation_errors_unconfirmed'))
            .addError(
                new Field.Error(
                    'confirmationsLimitReached',
                    '%field_human-confirmation_errors_confirmationsLimitReached'
                )
            );

        this.question = new QuestionField();
        this.userQuestion = new UserQuestionField();
        this.answer = new AnswerField();
        this.captcha = new CaptchaField();
        this.phoneConfirm = new PhoneConfirmField();

        this.setPhoneOptions = function(options) {
            self.phoneConfirm.setOptions(options);
            return self;
        };

        this.setPhonePlaceholder = function(placeholder) {
            self.phoneConfirm.setPlaceholder(placeholder);
            return self;
        };
    },

    /**
     * Field is empty when it's value is not 'phone' or 'captcha'
     * @param {Object} formData Posted form data
     */
    isEmpty: function(formData) {
        return ['phone', 'captcha'].indexOf(this._parseValue(formData)) === -1;
    },

    /**
     * When field is empty, set it's value to 'phone' as a default and show an error
     */
    onEmpty: function() {
        this.setValue('phone');
        this.getErrorByCode('unconfirmed').setActive();
    },

    _getPhoneForm: function() {
        if (this.getOption('captchaExtracted')) {
            return new Form(this.phoneConfirm.setRequired(), this.captcha.setRequired());
        } else {
            return new Form(this.phoneConfirm.setRequired());
        }
    },

    _getCaptchaForm: function() {
        var form;

        if (this.getOption('captchaAsync')) {
            form = new Form(
                this.question.setRequired(),
                this.userQuestion.setRequired(),
                this.answer.setRequired(),
                this.captcha.setOptional()
            );
        } else {
            form = new Form(
                this.question.setRequired(),
                this.userQuestion.setRequired(),
                this.answer.setRequired(),
                this.captcha.setRequired()
            );
        }

        return form;
    },

    /**
     * Validate the child blocks depending on the value of the switch.
     *
     * Granted the field is not empty,
     * validate phone-confirm if switch is in 'phone'
     * or question, user-question, answer and captcha if switch is in 'captcha'
     * @param {Object}          formData    Posted form data
     * @param {Passport.api}    api         Passport api instance
     */
    validate: function(formData, api) {
        var form;

        if (this._parseValue(formData) === 'phone') {
            form = this._getPhoneForm();
        } else {
            form = this._getCaptchaForm();
        }

        var that = this;

        form.setApi(api);
        return form.validate(formData).then(function(isValid) {
            if (!isValid) {
                that.getErrorByCode('unconfirmed').setActive();
                return ['unconfirmed'];
            }

            return [];
        });
    },

    onValid: function(formData) {
        this.setValue(this._parseValue(formData));
    },

    onInvalid: function(errors, formData) {
        this.setValue(this._parseValue(formData));
        this.setErrorsActive(errors);
    },

    compile: function(lang, api) {
        if (this.getValue()) {
            this.setOption('value', this.getValue());
        }

        if (this.getOption('captchaAsync')) {
            this.captcha.setOption('asyncCheck', true);
        }

        if (this.getOption('confirmed_phone_number')) {
            this.phoneConfirm.setValue(this.getOption('confirmed_phone_number'));
            this.phoneConfirm.setOption('state', 'confirmed');
            this.phoneConfirm.setOptional();
        }

        //Не нужно показывать эту ошибку, нужно показывать ошибки саб-блоков
        this.getErrorByCode('unconfirmed').setInactive();
        var compiled = this.__base.call(this, lang);

        var promises = [];
        var names = [];
        var fields = {
            phone_confirm: this.phoneConfirm
        };

        if (!this.getOption('phoneOnly')) {
            fields.question = this.question;
            fields.user_question = this.userQuestion;
            fields.answer = this.answer;
            fields.captcha = this.captcha;
        }

        _.each(fields, function(field, name) {
            promises.push(field.compile(lang, api));
            names.push(name);
        });

        var limitsPromise = api
            .params('track_id')
            .then(function(result) {
                return api.getRegistrationLimits({
                    track_id: result.body && result.body.id
                });
            })
            .then(function(result) {
                if (result.sms_remained_count < 1) {
                    // TODO: Пишем напрямую, т.к. объект уже скомпилен
                    _.assign(compiled.options, {
                        value: 'captcha',
                        'no-switch': true
                    });
                }
            });

        var fieldsPromise = when.all(promises).then(function(results) {
            _.assign(compiled, _.zipObject(names, results));
        });

        return when.all([limitsPromise, fieldsPromise]).then(function() {
            return compiled;
        });
    },

    normalize: function(formData) {
        var form = this.getValue() === 'captcha' ? this._getCaptchaForm() : this._getPhoneForm();

        return _.extend(form.normalize(formData), this.__base.call(this, formData));
    }
});

module.exports = HumanConfirmationFieled;
