(function() {
    passport.block('human-confirmation', 'control', {
        updateTimeout: 0, //Disable watching

        init: function() {
            this.$phoneGroup = this.$('.human-confirmation_phone, .human-confirmation-via-captcha');
            this.$captchaGroup = this.$('.human-confirmation_captcha, .human-confirmation-via-phone');
            this.$captchaBlock = this.$('.block_name_captcha');
            this.$switch = this.$('.human-confirmation-switch-wrap');

            this.inited.done(this.onInit);
        },

        onInit: function(that) {
            that.showSwitch();

            if (that.options.captchaExtracted) {
                that.extractCaptcha();
            }

            if (that.options.value === 'captcha') {
                that.switchToCaptcha();
                that.hintUpdated();
            } else {
                that.switchToPhone(); //Initialize all the subblocks states
            }
        },

        /**
         * Validate the human-confirmation. It
         * * Checks the switch value: either 'phone' or 'captcha'
         * * Validates the children selected by the switch
         *
         * @param {boolean} [suppressError]
         * @param {boolean} [passive=false]
         * Whether the block should be actively checked, or validation value should be fetched from the store
         */
        validate: function(suppressError, passive) {
            var check = passive
                ? function(block) {
                      //Block is checked against validator
                      var deferred = new $.Deferred();

                      deferred.resolve(passport.validator.check(block));
                      return deferred;
                  }
                : function(block) {
                      //Block is actively validated
                      return block.validate(suppressError);
                  };

            var that = this;
            var resultDeferred = new $.Deferred();

            if (this.val() === 'phone') {
                check(passport.block('phone-confirm')).then(function(result) {
                    that.emit('validation', result);
                    resultDeferred.resolve(result);
                });
            } else {
                var validationDeferreds = [];

                [
                    passport.block('question'),
                    passport.block('user-question'),
                    passport.block('answer'),
                    passport.block('captcha')
                ].forEach(function(block) {
                    validationDeferreds.push(check(block));
                });

                $.when.apply(null, validationDeferreds).then(function() {
                    var anyFailed = Array.prototype.slice.call(arguments).some(function(result) {
                        return result === false;
                    });

                    that.emit('validation', !anyFailed);
                    resultDeferred.resolve(!anyFailed);
                });
            }

            return resultDeferred;
        },

        events: {
            'click .human-confirmation-via-phone': 'switchToPhone',
            'click .human-confirmation-via-captcha': 'switchToCaptcha',
            'click .human-confirmation__warning_couldnotsend': 'couldNotSendAction',

            'update.question': 'hintUpdated',
            'update.user-question': 'hintUpdated',
            'update.answer': 'hintUpdated',

            'validation.question': 'captchaRelatedUpdate',
            'validation.user-question': 'captchaRelatedUpdate',
            'validation.answer': 'captchaRelatedUpdate',
            'validation.captcha': 'captchaRelatedUpdate',

            'confirmationsLimitReached.phone-confirm': 'confirmationsLimitReached',
            'phoneConfirmed.phone-confirm': 'phoneConfirmed',
            'couldnotsend.phone-confirm': 'couldnotsend',
            'show.phone-confirm': 'show'
        },

        captchaRelatedUpdate: function() {
            //Call to passive validation after the validator received an event
            setTimeout(this.validate.bind(this, true, true), 1);
        },

        switchToPhone: function() {
            this.$phoneGroup.removeClass('g-hidden');
            this.$captchaGroup.addClass('g-hidden');
            this.val('phone');
            this.validate(true); //To revalidate the subblocks
        },

        switchToCaptcha: function() {
            this.$phoneGroup.addClass('g-hidden');
            this.$captchaGroup.removeClass('g-hidden');
            this.val('captcha');
            this.validate(true); //To revalidate the subblocks
            this.toggleCouldNotSendWarning(false); //Hide the warning
        },

        extractCaptcha: function() {
            this.$captchaBlock.appendTo(this.$el);
        },

        showSwitch: function() {
            if (!this.switchHidenForever && !this.options['no-switch']) {
                this.$switch.removeClass('g-hidden');
            }
        },

        hideSwitch: function(forever) {
            if (forever) {
                this.switchHidenForever = forever;
            }

            this.$switch.addClass('g-hidden');
        },

        /**
         * Checks whether hint related fields are empty.
         * Shows/hides the switch based on that
         */
        hintUpdated: function() {
            //Settimeout is there in order for user-question to handle the event first
            setTimeout(
                function() {
                    var tracked = [
                        passport.block('question'),
                        passport.block('user-question'),
                        passport.block('answer')
                    ];

                    if (
                        tracked.some(function(block) {
                            return block.isEmpty();
                        })
                    ) {
                        this.showSwitch();
                    } else {
                        this.hideSwitch();
                    }
                }.bind(this),
                1
            );
        },

        /**
         * Intercepts custom event from phone-confirm block
         */
        phoneConfirmed: function() {
            this.hideSwitch();
            this.validate(true);
        },

        /**
         * Intercepts custom event from phone-confirm block
         */
        confirmationsLimitReached: function() {
            this.hideSwitch(true); //Hide switch forever
            this.switchToCaptcha();
            this.error('confirmationsLimitReached');
        },

        /**
         * Intercepts custom event from phone-confirm block
         */
        couldnotsend: function() {
            this.error('couldnotsendsms');
        },

        /**
         * Intercepts custom event from phone-confirm block
         */
        show: function() {
            this.toggleCouldNotSendWarning(false); //Hide the warning
        },

        couldNotSendAction: function() {
            this.error();
            this.switchToCaptcha();
        },

        toggleCouldNotSendWarning: function(show) {
            this.$('.human-confirmation__warning_couldnotsend_wrap').toggleClass('g-hidden', !show);
        },

        error: function(errors) {
            errors = [].concat(errors);
            this.parent.error.call(this, errors);
            this.toggleCouldNotSendWarning(errors.indexOf('couldnotsendsms') > -1);
        }
    });
})();
