/* global i18n:true */
/* eslint no-undef: "error" */

(function() {
    passport.block('login', 'control', {
        currentFirstName: '',
        currentLastName: '',
        currentSurName: '',
        lastError: null,
        suggestionEnabled: false,
        suggestedLogins: [],
        errorWasSuppressed: null,

        init: function() {
            this.suggestionsBox = this.$('.login__suggest');
            this.loginValid = this.$('.login__ok');
            this.loginLoading = this.$('.login__loading');
            this.loginRequirements = this.$('.js-requirements');
            this.isLiteFlag = this.$('#is_lite');

            var that = this;

            passport.block('firstname').inited.done(function(firstname) {
                that.currentFirstName = firstname.val();
            });

            passport.block('lastname').inited.done(function(lastname) {
                that.currentLastName = lastname.val();
            });

            passport.block('surname').inited.done(function(lastname) {
                that.currentSurName = lastname.val();
            });

            // Временное решение для угонщика
            var login = passport.util.getUrlParam('login');

            if (login) {
                this.inited.done(function() {
                    login = login.split('@').shift();
                    that.enableSuggestion();
                    that.val(login);
                });
            }
        },

        val: function() {
            var res = this.parent.val.apply(this, arguments);

            if (typeof res === 'string') {
                return res.trim();
            }

            return res;
        },

        events: {
            'validation.login': 'loadSuggest',
            'update.firstname': 'trackFirstname',
            'update.lastname': 'trackLastname',
            'update.surname': 'trackSurname',
            'click .login__suggestedLogin': 'applySuggestion',
            'focus .p-control__input': 'enableSuggestion'
        },

        trackFirstname: function(event, change) {
            this.currentFirstName = change.shift();
            this.loadSuggest();
        },
        trackLastname: function(event, change) {
            this.currentLastName = change.shift();
            this.loadSuggest();
        },
        trackSurname: function(event, change) {
            this.currentSurName = change.shift();
            this.loadSuggest();
        },

        getValidationParams: function() {
            var params = {
                login: this.val()
            };

            if (this.isLiteFlag.val()) {
                params.is_lite = 1;
            }

            return params;
        },

        validate: function(suppressError) {
            this.errorWasSuppressed = suppressError;
            return this.parent.validate.apply(this, arguments);
        },

        validationCallback: function(data, suppressError) {
            if (data.status === 'ok') {
                this.validationResult(true, null, suppressError);
            } else {
                this.validationResult(false, this.getErrorCodeFromData(data[0]), suppressError);
            }

            return Boolean(data.status === 'ok');
        },

        getErrorCodeFromData: function(data) {
            var code;
            var controlName = this.$ctrl.attr('name');
            var prefix = data.slice(0, controlName.length);

            if (prefix === controlName) {
                code = data.slice(controlName.length + 1);
            } else {
                code = data.replace('.', '_');
            }

            return code;
        },

        /**
         * Loads and shows suggests
         *
         * @param {Object} [event]
         * @param {Boolean} [isValid]
         * @param {String|Null} [errorCode]
         * @returns {Deferred}
         */
        loadSuggest: function(event, isValid, errorCode) {
            if (typeof isValid === 'undefined') {
                isValid = passport.validator.check(this);
            }

            if (typeof errorCode === 'undefined') {
                errorCode = this.lastError;
            }

            var loaded = new $.Deferred();

            this.loginValid.toggleClass('g-hidden', !isValid);

            if (!isValid) {
                var parameters = {
                    track_id: passport.track_id,
                    language: passport.language
                };

                var firstName = this.currentFirstName.trim();

                if (firstName.length > 0) {
                    parameters.firstname = firstName;
                }

                var lastName = this.currentLastName.trim() || this.currentSurName.trim();

                if (lastName.length > 0) {
                    parameters.lastname = lastName;
                }

                var login = this.val().trim();

                if (login.length > 0) {
                    parameters.login = login;
                }

                var fullNameAvailable = Boolean(firstName) && Boolean(lastName);

                if (login.length > 0 || fullNameAvailable) {
                    var prevError = this.lastError; //this.error() overrides last error. Save it to restore afterwards

                    if (this.suggestionEnabled) {
                        //Load suggests silently, but don't show anything
                        if (!this.ignoreErrorFromSuggest) {
                            this.error(); //Hides all errors
                        }
                        this.loginValid.addClass('g-hidden');
                        this.loginLoading.removeClass('g-hidden');
                    }

                    this.prevSuggestRequest = passport.api.request('suggestV2', parameters, {
                        abortPrevious: true,
                        cache: true
                    });

                    this.prevSuggestRequest
                        .done(
                            function(data) {
                                if (!this.ignoreErrorFromSuggest) {
                                    if (Array.isArray(errorCode) && errorCode.length === 0) {
                                        errorCode = prevError; //Restore previous error
                                    }

                                    if (!this.errorWasSuppressed) {
                                        this.error(errorCode);
                                    }
                                }

                                // PASSP-10039 — Отображается сообщение о необходимости выбора логина
                                // сразу после перехода на страницу дорегистрации лайта
                                //
                                // Если prevError === null, мы первый раз дергаем
                                // валидацию, при ините контрола, не показываем
                                // сразу ошибку
                                if (prevError !== null) {
                                    this.error(errorCode);
                                }

                                if (data.logins && Array.isArray(data.logins)) {
                                    this.suggestedLogins = data.logins;
                                    loaded.resolve(this.suggestedLogins);
                                    this.showSuggest();
                                }
                            }.bind(this)
                        )
                        .always(
                            function() {
                                this.loginLoading.addClass('g-hidden');
                                loaded.reject();
                            }.bind(this)
                        );
                } else {
                    loaded.reject();
                }
            } else {
                this.suggestionsBox.addClass('g-hidden');
                loaded.reject();
            }

            return loaded;
        },

        enableSuggestion: function() {
            this.suggestionEnabled = true;
            if (!passport.validator.check(this)) {
                this.showSuggest();
            }
        },

        showSuggest: function() {
            if (this.isLiteFlag.val()) {
                this.suggestionsBox.addClass('g-hidden');
                return;
            }

            if (!this.suggestionEnabled || this.suggestedLogins.length === 0) {
                this.suggestionsBox.addClass('g-hidden');
                return;
            }

            var list = '';

            this.suggestedLogins.forEach(function(login) {
                list +=
                    "<li class='login__singleSuggestion'><span class='login__suggestedLogin'>" + login + '</span></li>';
            });

            this.suggestionsBox
                .empty()
                .html(list)
                .removeClass('g-hidden');
        },

        applySuggestion: function(event) {
            this.val($(event.target).html());
        },

        error: function(errorCode) {
            if (Array.isArray(errorCode)) {
                errorCode = this.isLiteFlag.val() || errorCode.length > 0 ? errorCode[0] : '';
            }
            var errorsWithSpecialHint = ['domain_missing', 'native'];
            var errorsWithoutHint = ['missingvalue', 'notavailable', 'invalid', 'domain_invalid_type'];

            this.lastError = errorCode;
            this.parent.error.call(this, errorCode);

            if (errorsWithoutHint.indexOf(errorCode) !== -1) {
                this.loginValid.addClass('g-hidden');
                this.loginRequirements.addClass('g-hidden');
            } else if (errorsWithSpecialHint.indexOf(errorCode) !== -1) {
                this.setErrorHint(errorCode);
            } else if (!this.isLiteFlag.val()) {
                this.loginRequirements.toggleClass('g-hidden', !errorCode);
                this.loginRequirements.text(i18n('%login_requirements'));
            }
        },

        setErrorHint: function(errorCode) {
            var errorHints = {
                domain_missing: i18n('%login_email-example-hint'),
                native: i18n('%login_email-native-hint')
            };

            this.loginValid.addClass('g-hidden');
            this.loginRequirements.removeClass('g-hidden');
            this.loginRequirements.text(errorHints[errorCode]);
        }
    });
})();
