(function(exports) {
    var code = passport.block('phone-confirm-code', 'control', {
        isRequired: true,
        needsServerValidation: false,

        controlSelector: '.code_entry',

        nbErrorParams: {
            how: {
                at: 'bottom',
                my: 'top'
            }
        },

        state: 'invalid',

        getHandleID: function() {
            return 'code';
        },

        validate: function() {
            this.emit('validation', true);
        },

        validationHandle: function() {
            if (!this.options || !this.options.alternative) {
                return;
            }

            var val = Boolean(this.val().length);

            this.$('.js-confirm-code').toggleClass('_nb-action-button', val);
            this.state = val ? 'valid' : 'invalid';
        },

        /**
         * @param {PhoneModel} model
         */
        setModel: function(model) {
            this.model = model;
        },

        /**
         * @returns {PhoneModel}
         */
        getModel: function() {
            return this.model;
        },

        show: function() {
            var model = this.getModel();

            if (!model) {
                throw new Error('Model should be set');
            }

            if (model.isConfirmed()) {
                this.emit('phoneConfirmed');
            } else {
                this.error(); // Hide errors
                this.val('');

                this.updateDestinationPhone(model.getSanitized());

                if (this.options && this.options.resendTimer) {
                    this.startTimer();
                } else {
                    this.showResendButton();
                }

                this.$el.removeClass('g-hidden');

                // Show the password field if the phone requires a confirmation
                // eslint-disable-next-line max-len
                // @see https://beta.wiki.yandex-team.ru/passport/python/api/bundle/phone_v2/#dopolnitelnyekljuchipolozhitelnogootveta
                this.togglePasswordVisible(model.requiresPassword());

                this.focus();
            }
        },

        togglePasswordVisible: function(show) {
            this.$('.phone-confirm-password').toggleClass('g-hidden', !show);
        },

        hide: function() {
            this.$el.addClass('g-hidden');
        },

        focus: function() {
            this.$ctrl.focus();
        },

        updateDestinationPhone: function(phone) {
            this.$('.selected_phone').text(phone);
            this.$('.phone-confirm-code-fake-entry .p-control__input').val(phone);
        },

        events: {
            'click .submit_code': 'confirmCode',
            'keypress .p-control__input': '_confirmCodeOnEnter',
            'click .resend_code': 'retrySms',
            'click .back': 'back',
            'validation.phone-confirm-code': 'validationHandle',
            'click .selected_phone': 'backToEditPhone',
            'codeSent.phone-confirm': 'onCodeSent',
            'callsLimitReached.phone-confirm': 'onCallsLimit',
            'click .js-retry-by-sms': 'retrySms',
            'click .js-retry-by-call': 'retryCall',
            'blur .code_entry': 'onBlur'
        },

        onCallsLimit: function() {
            this.removeCallConfirmatioPlaceholders();
            this.showResendButton();
        },

        onCodeSent: function() {
            var isValidForCall = this.getModel().isValidForCall;

            this.toggleCallConfirmatioPlaceholders(isValidForCall);

            if (isValidForCall) {
                this.checkCallStatus();
            } else {
                this.removeCallConfirmatioPlaceholders();
            }

            this.error();
        },

        onCouldNotSend: function() {
            if (this.getModel().isValidForCall) {
                this.showResendButton();
                this.hideResendSMSButton();
            }
        },

        onCallError: function() {
            this.getModel().setValidForCall(false);
            this.preventTimer();
            this.removeCallConfirmatioPlaceholders();
            this.showResendButton();
            this.error('couldnotcall');
        },

        checkCallStatus: function() {
            var self = this;

            this.getModel()
                .checkCallStatus()
                .done(function(response) {
                    var errors = response.errors;

                    if (response.status !== 'error') {
                        return;
                    }

                    if (Array.isArray(errors)) {
                        if (errors.includes('call_confirm.not_ready')) {
                            setTimeout(function() {
                                self.checkCallStatus();
                            }, 300);
                        }

                        if (errors.includes('call_confirm.too_late') || errors.includes('track.invalid_state')) {
                            self.onCallError();
                        }
                    }
                })
                .fail(function() {
                    self.onCallError();
                });
        },

        onBlur: function() {
            if (!this.isEmpty()) {
                this.emit('onEntry');
            }
        },

        _confirmCodeOnEnter: function(event) {
            if (passport.util.getKeyCode(event) === 13) {
                this.$el.find('.submit_code').click();
                event.preventDefault();
            }
        },

        confirmCode: function(event) {
            if (event) {
                event.preventDefault();
            }

            var that = this;

            if (!this._capthcaRequired) {
                this._confirmCode();
                return;
            }

            passport
                .block('captcha')
                .validate()
                .done(function(valid) {
                    if (valid) {
                        that._capthcaRequired = false;
                        that._confirmCode();
                    }
                });
        },

        _confirmCode: function() {
            var that = this;
            var model = this.getModel();

            if (!model) {
                throw new Error('Model should be set');
            }

            if (model.getConfirmationsLeft() === 0) {
                this.emit('confirmationsLimitReached');
            } else {
                var code = $.trim(this.val());
                var password = this.$('.p-control__input_name_phone-confirm-password').val();
                var confirmBtn = this.$('.js-confirm-code');

                if (!code) {
                    this.error('missingvalue');
                } else if (model.requiresPassword() && !password) {
                    this.error('missingpassword');
                } else {
                    confirmBtn.addClass('is-disabled');
                    model
                        .confirmCode(code, password)
                        .done(function(result) {
                            if (result === true) {
                                that.emit('phoneConfirmed');
                            }
                            confirmBtn.removeClass('is-disabled');
                        })
                        .fail(function(errors) {
                            var containsError = function(err) {
                                return Array.isArray(errors) ? errors.indexOf(err) > -1 : errors === err;
                            };

                            if (containsError('captcha.required')) {
                                that.enableCaptcha();

                                if (containsError('password.not_matched')) {
                                    that.error('passwordnotmatched');
                                }
                            } else if (containsError('code.invalid')) {
                                that.error('codeinvalid');

                                if (model.requiresPassword()) {
                                    that.togglePasswordVisible(false);
                                    that.hideCaptha();
                                }
                            } else if (containsError('password.not_matched')) {
                                that.error('passwordnotmatched');
                            } else if (containsError('password.required')) {
                                that.togglePasswordVisible(true);
                                that.error('missingpassword');
                            } else if (containsError('account.global_logout')) {
                                that.error('global_logout');
                            } else {
                                if (that.options && that.options.hasConfirmedPhone) {
                                    that.error('try_later');
                                } else {
                                    that.error('generic');
                                }
                            }
                            confirmBtn.removeClass('is-disabled');
                        });
                }
            }
        },

        retryCall: function() {
            this.emit('retryCall');
            this.sendCode(null, true);
        },

        retrySms: function() {
            this.emit('retrySms');
            this.sendCode(null, false);
        },

        getDefaultPlaceholder: function() {
            return this.options.showCodePlaceholder ? i18n('%phone-confirm_code_label') : '';
        },

        removeCallConfirmatioPlaceholders: function() {
            this.$('.js-retry-by-call').remove();
            this.$('.js-phone-confirm-by-call').remove();
            this.$('.js-code-entry').attr({placeholder: this.getDefaultPlaceholder()});
        },

        toggleCallConfirmatioPlaceholders: function(tryToCall) {
            this.$('.js-code-entry').attr({
                placeholder: tryToCall ? i18n('%phone-confirm_call_code_label') : this.getDefaultPlaceholder()
            });

            this.$('.js-phone-confirm-by-call').toggleClass('g-hidden', !tryToCall);
        },

        sendCode: function(event, tryToCall) {
            if (event) {
                event.preventDefault();
            }

            var that = this;
            var model = this.getModel();

            if (typeof tryToCall !== 'undefined') {
                this.toggleCallConfirmatioPlaceholders(tryToCall);
                model.setValidForCall(tryToCall);
            }

            if (!model) {
                throw new Error('Model should be set');
            }

            if (model.getSendingsLeft() === 0) {
                this.emit('sendingLimitReached');
                this.hideResendButton();
            }

            this.startSendingFeedback();
            model
                .sendCode()
                .done(function() {
                    that.onCodeSent();
                    that.stopSendingFeedback();
                    that.emit('codeSent', null, tryToCall);
                })
                .fail(function(err) {
                    that.stopSendingFeedback(true);

                    if (err.indexOf('calls_limit.exceeded') > -1) {
                        that.emit('callsLimitReached');
                        that.onCallsLimit();
                        return;
                    }

                    that.onCouldNotSend();
                    that.emit('couldNotSend');
                });
        },

        /**
         * To make sure the spinner works for at least one second,
         * I keep the promise, that resolves some time after the startSendingFeedback was called
         */
        _sendingFeedbackFakePromise: null,

        /**
         * Start the visual feedback about the code being sent
         */
        startSendingFeedback: function() {
            this.$('.resend_code').addClass('g-hidden');

            passport.util.reattach(this.$('.resend_code_spinner').removeClass('g-hidden'));

            var deferred = new $.Deferred();

            setTimeout(deferred.resolve, 1000);
            this._sendingFeedbackFakePromise = deferred;
        },

        /**
         * Stop the visual feedback about the code being sent
         * @param {boolean} [sendingFailed]   Whether the sending failed
         */
        stopSendingFeedback: function(sendingFailed) {
            var stop = function() {
                this.$('.resend_code').removeClass('g-hidden');

                passport.util.reattach(this.$('.resend_code_spinner').addClass('g-hidden'));

                if (!sendingFailed) {
                    this.startTimer();
                } else {
                    this.hideResendButton();
                    this.hideResendTimer();
                }
            }.bind(this);

            if (!sendingFailed && this._sendingFeedbackFakePromise) {
                this._sendingFeedbackFakePromise.done(function() {
                    stop();
                });
            } else {
                stop();
            }
        },

        preventTimer: function() {
            this.showResendButton();
            this.hideResendTimer();
            this.stopTimer();
            this.$('.js-phone-confirm-by-call').addClass('g-hidden');
        },

        hideResendButton: function() {
            this.$('.resend_code_wrap').addClass('g-hidden');
        },

        showResendButton: function() {
            this.$('.resend_code_wrap').removeClass('g-hidden');
            this.$('.js-retry-by-call').removeClass('g-hidden');
            this.$('.js-retry-by-sms').removeClass('g-hidden');
        },

        hideResendTimer: function() {
            this.$('.js-resend-code-limit').addClass('g-hidden');
        },

        showResendTimer: function() {
            this.$('.js-resend-code-limit').removeClass('g-hidden');
        },

        hideResendSMSButton: function() {
            this.$('.js-retry-by-sms').addClass('g-hidden');
        },

        hideRecallButton: function() {
            this.$('.js-retry-by-call').addClass('g-hidden');
        },

        startTimer: function() {
            var timerText = this.$('.js-resend-seconds');
            var self = this;

            var model = this.getModel();
            var seconds = model.codeResendTimeout;

            timerText.text(seconds);

            this.hideResendButton();
            this.showResendTimer();
            this.timer = setInterval(function() {
                --seconds;

                if (seconds === 0) {
                    self.preventTimer();
                } else {
                    timerText.text(seconds < 10 ? '0' + seconds : seconds);
                }
            }, 1000);
        },

        stopTimer: function() {
            if (this.timer) {
                clearInterval(this.timer);
            }
        },

        backToEditPhone: function(event) {
            if (!this.options || !this.options.alternative) {
                return;
            }

            this.back(event);
        },

        back: function(event) {
            if (event) {
                event.preventDefault();
            }
            this.stopTimer();
            this.emit('restart');
        },

        hideCaptha: function() {
            this.$('.js-phone-confirm-captcha').addClass('g-hidden');
        },

        enableCaptcha: function() {
            var captcha = passport.block('captcha');

            if (this.$('.js-phone-confirm-captcha').length) {
                captcha.val('');
                captcha.enableControl();
                captcha.getNewCode();

                this._capthcaRequired = true;
                this.$('.js-phone-confirm-captcha').removeClass('g-hidden');
            } else {
                this.error('limit_exceeded');
            }
        }
    });

    exports.code = code;
    // eslint-disable-next-line no-undef
})(exports);
