(function() {
    var exports = {};
    /* borschik:include:phone-model/phone-model.js */
    /* borschik:include:phone-model/phone-manager.js */
    /* borschik:include:entry/entry.js */
    /* borschik:include:code/code.js */
    /* borschik:include:acknowledgement/acknowledgement.js */

    var PhoneModel = exports.PhoneModel;
    var PhoneManager = exports.PhoneManager;

    var entry = exports.entry;
    var code = exports.code;
    var acknowledgement = exports.acknowledgement;

    /**
     * Creates a phone-confirm view
     *
     * That view starts with phone entry.
     * Phone entry extends the phone view and validates the phone on its own.
     *
     * Once the phone entry got and validated the phone, phoneModel is created with that number.
     * Model is passed to code entry view and acknowledgement view.
     *
     * Code entry then is responsible for confirming the phone in that model.
     * Once the model is confirmed, the acknowledgement is shown.
     *
     * If user decides to enter another number, phone entry is shown once again.
     * The model is not destroyed unless user changes the number.
     *
     */
    passport.block('phone-confirm', 'control', {
        watchSilent: true,
        subBlocks: {
            entry: entry,
            code: code,
            acknowledgement: acknowledgement
        },

        _forEveryBlock: function(callback) {
            for (var blockName in this.subBlocks) {
                if (this.subBlocks.hasOwnProperty(blockName)) {
                    callback.call(this, this.subBlocks[blockName]);
                }
            }
        },

        _whenSubblocksInited: function(callback) {
            $.when
                .apply(
                    null,
                    [this.inited].concat(
                        $.map(this.subBlocks, function(block) {
                            return block.inited;
                        })
                    )
                )
                .done(callback.bind(this));
        },

        /**
         * Current state
         * entry || code || acknowledgement
         * @type: {string}
         */
        viewStage: 'entry',

        /**
         * Phone model instance for the currently entered phone
         * @type: {PhoneModel}
         */
        currentPhone: null,

        /**
         * @type: {PhoneModel}
         * @constructor
         */
        PhoneModel: PhoneModel,

        /**
         * @type {PhoneManager}
         * @constructor
         */
        PhoneManager: PhoneManager,

        init: function(options) {
            var that = this;

            this._forEveryBlock(function(block) {
                passport.mixins.interceptEvents(that, block, [
                    'restart',
                    'phoneEntered',
                    'phoneConfirmed',
                    'couldNotSend',
                    'sendingLimitReached',
                    'confirmationsLimitReached',
                    'callsLimitReached',
                    'retryCall',
                    'retrySms'
                ]);
            });

            this.phoneManager = new this.PhoneManager(this.PhoneModel, Boolean(options && options.prevCheckCaptcha));

            if (options && typeof this.phoneManager.mode[options.mode] === 'function') {
                // Set the phoneManager working mode (alters the backend handles used)
                this.phoneManager.mode[options.mode]();
            }

            if (options && options.hasConfirmedPhone) {
                this.subBlocks.entry.inited.done(function(entry) {
                    var options = entry.options || (entry.options = {});

                    options.noValidation = true;
                });
            }

            if (options && options.state === 'confirmed') {
                that.emit('phoneConfirmed');
                this._whenSubblocksInited(function() {
                    var phone = that.subBlocks.entry.val();

                    if (phone.length > 0) {
                        var model = that.phoneManager.getModel(phone);

                        model.setSanitized(phone).setConfirmed();
                        that.setModel(model);
                        that.show('acknowledgement');
                    }
                });
            }

            if (options && options.popupErrors) {
                // Error override to show errors in a popup if such option is given
                var errPopup = nb.block(this.$('.js-phone-confirm-general-errors-popup').get(0));
                var originalError = this.error;

                this.error = function(code) {
                    this._forEveryBlock(function(block) {
                        block.error();
                    });

                    this.nbctrl = undefined; // Phone-confirm has no control of it's own, no need
                    originalError.apply(this, arguments);

                    if (code) {
                        if (!errPopup.isOpen()) {
                            errPopup.open({
                                where: this.$el,
                                how: {
                                    at: 'right',
                                    my: 'left',
                                    autoclose: false
                                }
                            });
                        }
                    } else {
                        errPopup.close();
                    }
                };
            }

            if (!this.options || !this.options.hasConfirmedPhone) {
                this.emit('showEntry');
            }
        },

        events: {
            'phoneConfirmed.phone-confirm': 'setConfirmedState'
        },

        /**
         * Whether this control is required
         * @private
         */
        isRequired: true,

        /**
         * Toggle this control required state
         * @param {boolean} required
         */
        toggleRequired: function(required) {
            // TODO: test
            required = Boolean(required);
            this.isRequired = required;
            this._forEveryBlock(function(block) {
                block.isRequired = required;
            });
            this.validate(true);
        },

        validate: function(suppressError) {
            var deferred = new $.Deferred();

            if (!this.inited.already) {
                deferred.resolve(false);
                return deferred;
            }

            var valid = true; // Always valid when optional

            if (this.isRequired) {
                if (!this.skipEntryValidation && !this.options.hasConfirmedPhone) {
                    this.subBlocks.entry.validate(suppressError);
                }

                // Valid when there is a phone and it is confirmed
                var activePhone = this.phoneManager.getActive();

                valid = Boolean(activePhone && activePhone.isConfirmed());
            }

            this.emit('validation', valid);
            if (!suppressError) {
                this.error(valid ? null : 'needsconfirmation');
            }

            deferred.resolve(valid);
            return deferred.promise();
        },

        restart: function() {
            if (this.options && this.options.hasConfirmedPhone) {
                return;
            }

            this.show('entry');
        },

        phoneEntered: function(number, isValidForCall) {
            var that = this;

            if (this.options && this.options.prevCheckCaptcha) {
                var isCaptchaChecked = passport.validator.check(passport.block('captcha'));

                if (!isCaptchaChecked) {
                    passport
                        .block('captcha')
                        .validate()
                        .done(function(result) {
                            if (!result) {
                                that.invalidCaptcha();
                                return;
                            }

                            that.phoneEntered(number);
                        });

                    return;
                }
            }

            this.setModel(this.phoneManager.getModel(number));

            this.getModel().setValidForCall(isValidForCall);

            if (this.getModel().isCodeSent()) {
                this.show(this.getModel().isConfirmed() ? 'acknowledgement' : 'code');
                this.emit('codeSent', number, isValidForCall);
            } else {
                this.startSendingFeedback();
                this.getModel()
                    .sendCode(this.options)
                    .done(function() {
                        that.onSendCode(number);
                    })
                    .fail(function(err) {
                        that.stopSendingFeedback(true);

                        var containsError = function(code) {
                            return Array.isArray(err) ? err.indexOf(code) > -1 : err === code;
                        };

                        if (containsError('phone.confirmed')) {
                            that.phoneConfirmed();
                            return;
                        }

                        if (containsError('captcha.not_checked')) {
                            that.invalidCaptcha();
                            return;
                        }

                        if (
                            containsError('phone_secure.not_found') ||
                            containsError('phone_secure.not_match') ||
                            containsError('phone.not_matched')
                        ) {
                            that.invalidSecurePhone();
                            return;
                        }

                        if (containsError('rate.limit_exceeded')) {
                            that.limitExceeded();
                            return;
                        }

                        if (containsError('phone.compromised')) {
                            that.phoneCompromised();
                            return;
                        }

                        if (containsError('phone_secure.bound_and_confirmed')) {
                            that.phoneBoundAndConfirmed();
                            return;
                        }

                        if (containsError('account.global_logout')) {
                            that.error('global_logout');
                            return;
                        }

                        if (containsError('calls_limit.exceeded')) {
                            that.callsLimitReached();
                            return;
                        }

                        if (err && err.status >= 400 && isValidForCall) {
                            that.phoneEntered(number, false);
                            return;
                        }

                        that.couldNotSend();
                    });
            }
        },

        onSendCode: function(number) {
            this.stopSendingFeedback();
            this.show('code');
            this.emit('codeSent', number, this.getModel().isValidForCall);
        },

        /**
         * To make sure the spinner works for at least one second,
         * I keep the promise, that resolves some time after the startSendingFeedback was called
         */
        _sendingFeedbackFakePromise: null,

        /**
         * Start the visual feedback about the code being sent
         */
        startSendingFeedback: function() {
            this.subBlocks.entry.startSendingFeedback();

            var deferred = new $.Deferred();

            setTimeout(deferred.resolve, 1000);
            this._sendingFeedbackFakePromise = deferred;
        },

        /**
         * Stop the visual feedback about the code being sent
         * @param {boolean} [sendingFailed]   Whether the sending failed
         */
        stopSendingFeedback: function(sendingFailed) {
            var that = this;

            if (!sendingFailed && this._sendingFeedbackFakePromise) {
                this._sendingFeedbackFakePromise.done(function() {
                    that.subBlocks.entry.stopSendingFeedback(!sendingFailed);
                    if (that.options.hasConfirmedPhone) {
                        that.subBlocks.entry.disableControl();
                    }
                });
            } else {
                that.subBlocks.entry.stopSendingFeedback();
                if (that.options.hasConfirmedPhone) {
                    that.subBlocks.entry.disableControl();
                }
            }
        },

        /**
         * @param {PhoneModel} model
         */
        setModel: function(model) {
            this.currentPhone = model;
            this.subBlocks.code.setModel(model);
            this.subBlocks.acknowledgement.setModel(model);
        },

        /**
         * @returns {PhoneModel}
         */
        getModel: function() {
            return this.currentPhone;
        },

        phoneConfirmed: function() {
            this.show('acknowledgement');
            this.emit('phoneConfirmed');
        },

        show: function(stage) {
            this._forEveryBlock(function(block) {
                block.hide();
            });

            switch (stage) {
                case 'entry':
                    this.emit('showEntry');
                    entry.show();
                    break;
                case 'code':
                    this.emit('showCode');
                    code.show();
                    break;
                case 'acknowledgement':
                    this.emit('showAcknowledgement');
                    acknowledgement.show();
                    break;
                default:
                    throw new Error('Unknown stage');
            }

            this.onShow(stage);

            if (this.options && this.options.popupErrors) {
                // Hide all the errors on sub-blocks
                this._forEveryBlock(function(block) {
                    block.error();
                });

                // Redraw all the errors due to possible position change
                $(window).trigger('resize');
            }
        },

        onShow: function(stage) {
            this.error();
            this.validate(true);
            this.emit('show', stage);
            this.viewStage = stage;
        },

        val: function() {
            return this.subBlocks.entry.val.apply(this.subBlocks.entry, arguments);
        },

        focus: function() {
            this.subBlocks.entry.$ctrl.focus();
        },

        updatePhoneMask: function() {
            this.subBlocks.entry.updateMask();
        },

        phoneCompromised: function() {
            this.emit('compromised');
            this.error('compromised');
        },

        phoneBoundAndConfirmed: function() {
            this.emit('bound_and_confirmed');
            this.error('bound_and_confirmed');
        },

        callsLimitReached: function() {
            this.emit('callsLimitReached');
            this.error('calls_limitexceeded');
        },

        couldNotSend: function() {
            this.emit('couldnotsend');
            this.error('couldnotsend');
        },

        sendingLimitReached: function() {
            this.emit('sendingLimitReached');
        },

        confirmationsLimitReached: function() {
            this.emit('confirmationsLimitReached');
            if (this.options && this.options.hasConfirmedPhone) {
                this.error('limitexceeded_one_number');
            } else {
                this.error('limitexceeded');
            }
        },

        invalidCaptcha: function() {
            this.emit('invalidcaptcha');
            this.error('invalidcaptcha');
        },

        invalidSecurePhone: function() {
            this.emit('invalidsecurephone');
            this.error('invalidsecurephone');
        },

        limitExceeded: function() {
            this.error('limitexceeded_one_number');
            this.emit('limitexceeded');
        },

        setConfirmedState: function() {
            this.$el.find('.phone-confirm-fake-input[name=phone-confirm-state]').val('confirmed');
        },

        retryCall: function() {
            this.error();
        },

        retrySms: function() {
            this.error();
        }
    });
})();
