(function(exports) {
    /**
     * Model describing the validation process of the phone number
     *
     * @param {string} unsanitizedNumber        Phone number as typed by the user
     * @constructor
     */
    var PhoneModel = function(unsanitizedNumber) {
        if (!unsanitizedNumber || typeof unsanitizedNumber !== 'string') {
            throw new Error('Unsanitized number should be a string');
        }

        /**
         * Flag determining if the code was sent to that phone
         * @type {boolean}
         * @private
         */
        this._codeSent = false;

        /**
         * Phone number entered by the user
         * @type {string}
         * @private
         */
        this._unsanitized = unsanitizedNumber;

        /**
         * International format for the phone number
         * @type {string}
         * @private
         */
        this._sanitized = null;

        /**
         * International format for the phone number without mask
         * @type {string}
         * @private
         */
        this._sanitizedUnmasked = null;

        /**
         * Flag determining whether the phone has been confirmed
         * @type {boolean}
         * @private
         */
        this._isConfirmed = false;

        /**
         * How many more times can the code be sent.
         * @type {number}
         * @private
         */
        this._sendingsLeft = 99; // These should be set according to api.

        /**
         * How many more times can the confirmation be attempted.
         * @type {number}
         * @private
         */
        this._confirmationsLeft = 99; // These should be set according to api.

        /**
         * Requests handler
         * @type {passport.api.request}
         * @private
         */
        this._request = passport.api.request;

        /**
         * Offset in ms between the server and local time
         *
         * offset = local time - server time
         * where both times are timestamps in ms from the epoch
         *
         * @type {number}
         * @private
         */
        this._time_offset = 0;

        /**
         * Timestamp when next message is allowed to be sent in ms from the epoch
         *
         * @type {null}
         * @private
         */
        this._deny_until = null;

        /**
         * Phone confirmation mode
         * Selects the backend handle
         *
         * @type {Mode}
         */
        this.mode = null;

        this.codeResendTimeout = 35;

        /**
         * Флаг для blocks.js
         * Говорит нужно ли проверить валидность капчи перед отправкой смс
         *
         * @type {Boolean}
         */
        this.checkCaptcha = false;

        /**
         * Флаг для blocks.js
         * Говорит можно ли подтверждать телефон звонком
         *
         * @type {Boolean}
         */
        this.isValidForCall = false;
    };

    PhoneModel.prototype = {
        setMode: function(mode) {
            if (!(mode instanceof PhoneModel.Mode)) {
                throw new Error('Mode expected');
            }

            this.mode = mode;
        },

        setCheckCaptchaMode: function(checkCaptcha) {
            this.checkCaptcha = checkCaptcha;
        },

        setValidForCall: function(isValidForCall) {
            this.isValidForCall = isValidForCall;
        },

        /**
         * Send a code to this number
         *
         * Handles sending, re-sending, timeouts and the number of retries
         *
         * @returns {$.Deferred}
         */
        sendCode: function() {
            if (this.getSendingsLeft() < 1) {
                throw new Error('Maximum allowed codes sent. No more retries allowed.');
            }

            var that = this;
            var deferred = new $.Deferred();

            if (this.isConfirmed()) {
                deferred.resolve();
                return deferred;
            }

            if (this.getTimeout()) {
                // If a request is already scheduled — return it
                if (this._scheduledDeferred) {
                    return this._scheduledDeferred.promise();
                }

                // Wait for the timeout
                this._scheduledDeferred = deferred;
                this._scheduledTimeout = setTimeout(function() {
                    that._sendCodeRequest()
                        .always(function() {
                            that._scheduledDeferred = null;
                        })
                        .done(function() {
                            deferred.resolve();
                        })
                        .fail(function(result) {
                            deferred.reject(result);
                        });
                }, this.getTimeout() + 1000);

                return deferred.promise();
            }

            return this._sendCodeRequest();
        },

        checkCallStatus: function() {
            var deferred = new $.Deferred();

            this._request('confirm-phone-check-status', {}, {abortPrevious: true})
                .done(function(data) {
                    deferred.resolve(data);
                })
                .fail(function(error) {
                    deferred.reject(error);
                });

            return deferred.promise();
        },
        /**
         * Sends a request to send the code
         *
         * @returns {$.Deferred.promise()}
         * @private
         */
        _sendCodeRequest: function() {
            var that = this;
            var deferred = new $.Deferred();
            var requestData = {
                number: this._unsanitized,
                mode: this.mode.get(),
                checkCaptcha: this.checkCaptcha,
                confirmMethod: this.isValidForCall ? 'by_call' : 'by_sms'
            };

            this._request('phone-confirm-code-submit', requestData, {abortPrevious: true})
                .done(function(data) {
                    if (data.status !== 'ok') {
                        data.errors = data.errors || [];
                        if (that._arrayWithSingleElement(data.errors, 'smslimit.exceeded')) {
                            that._sendingsLeft = 0;
                        }

                        if (that._arrayWithSingleElement(data.errors, 'phone.confirmed')) {
                            that._codeSent = true;
                            that._sanitized = data.number.masked_international || data.number.international || '';
                            that._sanitizedUnmasked = data.number.international || '';
                            that.setConfirmed();
                        }

                        deferred.reject(data.errors);
                    } else {
                        that._sendingsLeft--;

                        data.number = data.number || {};
                        that._sanitized = data.number.masked_international || data.number.international || '';
                        that._sanitizedUnmasked = data.number.international || '';
                        that._requiresPassword = data.is_password_required;

                        that._time_offset = $.now() - data.now;
                        that._deny_until = data.deny_resend_until * 1000; // Unix timestamp 2 js timestamp

                        that._codeSent = true;
                        that.codeResendTimeout = data.resend_timeout || 35;

                        deferred.resolve();
                    }
                })
                .fail(function(error) {
                    deferred.reject(error);
                });

            return deferred.promise();
        },

        /**
         * Confirm the code typed in by the user
         * @param {string}   code       Confirmation code
         * @param {string}  [password]  password, if it is necessary for confirmation
         * @returns {$.Deferred}
         */
        confirmCode: function(code, password) {
            if (this.getConfirmationsLeft() < 1) {
                throw new Error('Maximum allowed confirmations done. No more allowed.');
            }

            var omitEmpty = function(obj) {
                Object.keys(obj).forEach(function(key) {
                    if (!obj[key]) {
                        delete obj[key];
                    }
                });

                return obj;
            };

            var that = this;
            var deferred = new $.Deferred();
            var requestData = omitEmpty({
                code: code,
                password: password,
                mode: this.mode.get()
            });

            if (this.isConfirmed()) {
                deferred.resolve(true);
                return deferred;
            }

            this._request('phone-confirm-code', requestData, {abortPrevious: true})
                .done(function(data) {
                    if (data.status !== 'ok') {
                        data.errors = data.errors || [];

                        if (that._arrayWithSingleElement(data.errors, 'code.invalid')) {
                            that._confirmationsLeft--;
                        } else if (that._arrayWithSingleElement(data.errors, 'confirmations_limit.exceeded')) {
                            that._confirmationsLeft = 0;
                        }

                        deferred.reject(data.errors);
                    } else {
                        that._isConfirmed = true;
                        deferred.resolve(true);
                    }
                })
                .fail(function() {
                    deferred.reject();
                });

            return deferred.promise();
        },

        /**
         * Method to check if api returned a single expected error
         *
         * @param {array} arr
         * @param {string} code
         * @returns {boolean}
         * @private
         */
        _arrayWithSingleElement: function(arr, code) {
            return $.isArray(arr) && arr.length === 1 && arr[0] === code;
        },

        /**
         * Get phone entered by the user
         * @returns {string}
         */
        getUnsanitized: function() {
            return this._unsanitized;
        },

        /**
         * Get phone number sanitized by the backend
         * @returns {string|null}
         */
        getSanitized: function() {
            return this._sanitized;
        },

        /**
         * Get phone number sanitized by the backend without mask
         * @returns {string|null}
         */
        getSanitizedUnmasked: function() {
            return this._sanitizedUnmasked;
        },

        /**
         * Determine whether the phone has been successfully confirmed
         * @returns {boolean}
         */
        isConfirmed: function() {
            return this._isConfirmed;
        },

        setConfirmed: function() {
            this._isConfirmed = true;
            return this;
        },

        /**
         * Returns how many times can code be resent.
         * @returns {number}
         */
        getSendingsLeft: function() {
            return this._sendingsLeft;
        },

        /**
         * Returns how many times can the code be entered incorrectly before the confirmation is prohibited
         * @returns {number}
         */
        getConfirmationsLeft: function() {
            return this._confirmationsLeft;
        },

        /**
         * Returns a number of ms until the next code can be sent.
         * @returns {number}
         */
        getTimeout: function() {
            var timeout = this._deny_until - $.now() + this._time_offset;

            return timeout < 0 ? 0 : timeout;
        },

        /**
         * Set a timeout for the model
         * @param {number} timeout
         * @retuns {PhoneModel}
         */
        setTimeout: function(timeout) {
            if (typeof timeout !== 'number') {
                throw new Error('Timeout should be a number');
            }

            this._deny_until = $.now() + timeout;
            return this;
        },

        /**
         * Abort sending the request
         */
        clearTimeout: function() {
            clearTimeout(this._scheduledTimeout);
            if (this._scheduledDeferred) {
                this._scheduledDeferred.resolve();
            }
        },

        /**
         * Whether the code had been sent for the model
         * @returns {boolean}
         */
        isCodeSent: function() {
            return this._codeSent;
        },

        requiresPassword: function() {
            return this._requiresPassword;
        }
    };

    ['request', 'sendingsLeft', 'confirmationsLeft', 'sanitized'].forEach(function(field) {
        PhoneModel.prototype['set' + passport.util.capitalize(field)] = function(newValue) {
            this['_' + field] = newValue;
            return this;
        };
    });

    PhoneModel.Mode = function() {
        this.confirm(); // By default
    };
    PhoneModel.Mode.prototype.confirm = function() {
        this._mode = 'confirm';
    };
    PhoneModel.Mode.prototype.confirmAndBindSecure = function() {
        this._mode = 'confirmAndBindSecure';
    };
    PhoneModel.Mode.prototype.tracked = function() {
        this._mode = 'tracked';
    };
    PhoneModel.Mode.prototype.trackedWithUpdate = function() {
        this._mode = 'trackedWithUpdate';
    };
    PhoneModel.Mode.prototype.confirmForRestore = function() {
        this._mode = 'confirmForRestore';
    };
    PhoneModel.Mode.prototype.restore = function() {
        this._mode = 'restore';
    };
    PhoneModel.Mode.prototype.restoreBind = function() {
        this._mode = 'restoreBind';
    };
    PhoneModel.Mode.prototype.get = function() {
        return this._mode;
    };

    exports.PhoneModel = PhoneModel;
    // eslint-disable-next-line no-undef
})(exports);
