var _ = require('lodash');
var when = require('when');

var Form = require('pform');
var Field = require('pform/Field');
var PhoneField = require('../phone-confirm/phone-confirm.field');
var LoginField = require('../login-preset/login-preset.field');
var LoginPresetField = require('../login-preset/login-preset.field');

/**
 * @extends Field
 */
var SwitchLoginPhone = require('inherit')(Field, {
    __constructor: function() {
        this._init('switch-login-phone');

        this.login = new LoginField();
        this.phone = new PhoneField().setOptions({
            mask: true,
            unmaskedAcknowledgement: true
        });
        this.loginPreset = new LoginPresetField();
    },

    /**
     * Field is empty when it's value is not 'phone' or 'login'
     * @param {Object} formData Posted form data
     */
    isEmpty: function(formData) {
        return ['phone', 'login', 'login-with-phone'].indexOf(this._parseValue(formData)) === -1;
    },

    /**
     * When field is empty, set it's value to 'login' as a default
     */
    onEmpty: function() {
        this.setValue('login');
    },

    _getPhoneForm: function() {
        return new Form(this.login.setRequired(), this.phone.setRequired());
    },

    _getLoginForm: function() {
        return new Form(this.login.setRequired());
    },

    /**
     * Validate the child blocks depending on the value of the switch.
     *
     * Granted the field is not empty,
     * validate phone if switch is in 'phone'
     * or login if switch is in 'login'
     * @param {Object}          formData    Posted form data
     * @param {Passport.api}    api         Passport api instance
     */
    validate: function(formData, api) {
        var form;

        if (this._parseValue(formData) === 'login') {
            form = this._getLoginForm();
        } else {
            form = this._getPhoneForm();
        }

        form.setApi(api);
        return form.validate(formData).then(function() {
            return [];
        });
    },

    onInvalid: function(errors, formData) {
        this.setValue(this._parseValue(formData));
        this.setErrorsActive(errors);
    },

    compile: function(lang, api) {
        if (!this.getValue()) {
            this.setValue('login');
        }

        var compiled = this.__base.call(this, lang);

        var promises = [];
        var names = [];

        _.each(
            {
                phone_confirm: this.phone,
                login_preset: this.login
            },
            function(field, name) {
                promises.push(field.compile(lang, api));
                names.push(name);
            }
        );

        return when.all(promises).then(function(results) {
            for (var i = 0; i < results.length; i++) {
                compiled[names[i]] = results[i];
            }

            return compiled;
        });
    },

    normalize: function(formData) {
        var form = this.getValue() === 'login' ? this._getLoginForm() : this._getPhoneForm();

        return _.extend(form.normalize(formData), this.__base.call(this, formData));
    }
});

module.exports = SwitchLoginPhone;
